<?php

namespace App\Livewire\Dashboard\Profile;

use Livewire\Component;
use App\Models\User\User;
use App\Models\User\UserProfile;
use App\Services\User\BusinessHourService;
use App\Enums\DayOfWeek;
use Illuminate\Support\Facades\Auth;
use Livewire\WithFileUploads;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\DB;
use Filament\Notifications\Notification;
use Livewire\Attributes\Layout;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Log;

#[Layout('layouts.dashboard.dashboard-app')]
class ProfileBasic extends Component
{
    use WithFileUploads;

    public ?string $first_name = null;
    public ?string $last_name = null;
    public ?string $company_name = null;
    public ?string $website = null;
    public ?string $email = null;
    public ?string $phone = null;
    public ?string $mobile = null;
    public ?int $year_established = null;
    public ?string $business_address = null;
    public ?string $business_timezone = null;
    public ?string $about = null;
    public ?string $slogan = null;
    public $avatar;
    public bool $isDirty = false;
    public ?array $social_media_links = [];
    public ?array $business_hours = [];
    public ?array $seasonal_hours = [];
    public ?array $profile_tags = [];

    // Social media link fields
    public ?string $facebook = null;
    public ?string $instagram = null;
    public ?string $twitter = null;
    public ?string $linkedin = null;
    public ?string $youtube = null;
    public ?string $tiktok = null;
    public ?string $pinterest = null;

    public int $about_length = 500;

    public User $user;
    public ?UserProfile $profile = null;

    public ?array $credentials = [];

    public ?array $awards_recognitions = [];

    public function mount()
    {
        $this->user = Auth::user();
        $this->profile = $this->user->profile;

        // Initialize form fields with existing data
        if ($this->profile) {
            $this->first_name = $this->profile->first_name;
            $this->last_name = $this->profile->last_name;
            $this->company_name = $this->profile->company_name;
            $this->website = $this->profile->website;
            $this->phone = $this->profile->phone;
            $this->mobile = $this->profile->mobile;
            $this->year_established = $this->profile->year_established;
            $this->about = $this->profile->about;
            $this->slogan = $this->profile->slogan;
            $this->social_media_links = $this->profile->social_media_links ?? [];
            $this->business_timezone = $this->profile->getMeta('business_timezone');

            // Load business hours from new table structure
            $businessHourService = app(BusinessHourService::class);
            $existingBusinessHours = $businessHourService->getBusinessHours($this->user);

            if ($existingBusinessHours->isEmpty()) {
                // If no business hours exist, check for legacy data and convert it
                $legacyBusinessHours = $this->profile->business_hours ?? [];
                if (!empty($legacyBusinessHours)) {
                    $existingBusinessHours = $businessHourService->convertLegacyBusinessHours($this->user, $legacyBusinessHours);
                } else {
                    // Create default business hours
                    $existingBusinessHours = $businessHourService->createDefaultBusinessHours($this->user);
                }
            }

            // Convert to array format for form
            $this->business_hours = [];
            foreach (DayOfWeek::getAllDays() as $day) {
                $dayHours = $existingBusinessHours->firstWhere('day_of_week', $day->value);
                $this->business_hours[$day->value] = [
                    'day_of_week' => $day->value,
                    'day_label' => $day->getLabel(),
                    'is_open' => $dayHours ? $dayHours->is_open : false,
                    'hours_label' => $dayHours ? ($dayHours->hours_label ?? '') : '',
                    'open_time' => $dayHours && $dayHours->open_time ? substr($dayHours->open_time, 0, 5) : '09:00',
                    'close_time' => $dayHours && $dayHours->close_time ? substr($dayHours->close_time, 0, 5) : '17:00',
                ];
            }


            $this->seasonal_hours = $this->profile->getMeta('seasonal_hours') ?? [];

            // Load profile tags from meta
            $this->profile_tags = $this->profile->getMeta('profile_tags') ?? [];

            // If seasonal_hours is a sequential array, convert to UUID keys
            if (is_array($this->seasonal_hours) && !empty($this->seasonal_hours) && array_is_list($this->seasonal_hours)) {
                $newSeasonalHours = [];
                foreach ($this->seasonal_hours as $hours) {
                    $uuid = (string) \Illuminate\Support\Str::uuid();
                    $newSeasonalHours[$uuid] = $hours;
                }
                $this->seasonal_hours = $newSeasonalHours;
            }

            // Initialize social media links from profile
            $this->facebook = $this->social_media_links['facebook'] ?? null;
            $this->instagram = $this->social_media_links['instagram'] ?? null;
            $this->twitter = $this->social_media_links['twitter'] ?? null;
            $this->linkedin = $this->social_media_links['linkedin'] ?? null;
            $this->youtube = $this->social_media_links['youtube'] ?? null;
            $this->tiktok = $this->social_media_links['tiktok'] ?? null;
            $this->pinterest = $this->social_media_links['pinterest'] ?? null;

            $this->credentials = $this->profile->credentials ?? [];

            // If credentials is an array without UUID keys, convert it to the right format
            if (is_array($this->credentials) && !empty($this->credentials) && array_is_list($this->credentials)) {
                $newCredentials = [];
                foreach ($this->credentials as $credential) {
                    $uuid = (string) \Illuminate\Support\Str::uuid();
                    if (is_string($credential)) {
                        $newCredentials[$uuid] = [
                            'credential_type' => \App\Enums\ProfessionalCredentionals::CertifiedBy->value,
                            'credential' => $credential
                        ];
                    } elseif (is_array($credential)) {
                        $newCredentials[$uuid] = [
                            'credential_type' => $credential['credential_type'] ?? \App\Enums\ProfessionalCredentionals::CertifiedBy->value,
                            'credential' => $credential['credential'] ?? ''
                        ];
                    }
                }
                $this->credentials = $newCredentials;
            }

            // Load awards from meta data only
            $this->awards_recognitions = $this->profile->getMeta('awards_recognitions') ?? [];

            // If awards is an array without UUID keys, convert it to the right format
            if (is_array($this->awards_recognitions) && !empty($this->awards_recognitions) && array_is_list($this->awards_recognitions)) {
                $newAwards = [];
                foreach ($this->awards_recognitions as $award) {
                    $uuid = (string) \Illuminate\Support\Str::uuid();
                    if (is_array($award) && isset($award['title'])) {
                        $newAwards[$uuid] = $award;
                    }
                }
                $this->awards_recognitions = $newAwards;
            }
        }

        $this->email = $this->user->email;
    }

    public function updatedAvatar()
    {
        $this->isDirty = true;
    }

    public function updated($field)
    {
        if ($field !== 'isDirty') {
            $this->isDirty = true;
        }
        $this->validateOnly($field);
    }

    public function deleteAvatar()
    {
        try {
            $media = $this->user->getFirstMedia('avatar');
            if ($media) {
                $media->delete();

                // Force a UI refresh
                $this->dispatch('avatar-updated');
                $this->isDirty = true;

                Notification::make()
                    ->title(__('Success!'))
                    ->success()
                    ->body(__('Profile picture has been removed successfully.'))
                    ->send();
            }
        } catch (\Exception $e) {
            Notification::make()
                ->title(__('Error!'))
                ->danger()
                ->body(__('Failed to remove profile picture. Please try again.'))
                ->send();
        }
    }


    public function addProfileTag($tag)
    {
        $tag = trim((string) $tag);
        if ($tag === '') {
            return;
        }
        // Enforce max tag length 20
        $tag = Str::limit($tag, 20, '');

        // Enforce max 3 tags and avoid duplicates
        if (count($this->profile_tags) >= 3) {
            Notification::make()
                ->title(__('Limit Reached'))
                ->warning()
                ->body(__('You can add a maximum of 3 profile tags.'))
                ->send();
            return;
        }
        if (!in_array($tag, $this->profile_tags)) {
            $this->profile_tags[] = $tag;
            $this->isDirty = true;
        }
    }

    public function removeProfileTag($index)
    {
        unset($this->profile_tags[$index]);
        $this->profile_tags = array_values($this->profile_tags);
        $this->isDirty = true;
    }


    public function addSeasonalHours()
    {
        if (count($this->seasonal_hours) >= 7) {
            Notification::make()
                ->title(__('Limit Reached'))
                ->warning()
                ->body(__('You can add a maximum of 7 seasonal hours entries.'))
                ->send();
            return;
        }

        // Generate a UUID as the key for the new seasonal hours entry
        $uuid = (string) \Illuminate\Support\Str::uuid();

        // Add a new seasonal hours entry with the UUID key
        $this->seasonal_hours[$uuid] = [
            'is_open' => true,
            'season_name' => '',
            'start_date' => date('Y-m-d'),
            'end_date' => date('Y-m-d', strtotime('+1 month')),
            'days' => '',
            'start_time' => '09:00',
            'end_time' => '17:00'
        ];

        $this->isDirty = true;
    }

    public function removeSeasonalHours($key)
    {
        // Directly remove the seasonal hours entry with the specific key
        unset($this->seasonal_hours[$key]);

        // Make sure we're working with the actual array, not references
        $this->seasonal_hours = array_filter($this->seasonal_hours);

        $this->isDirty = true;
    }

    public function addCredential()
    {
        if (count($this->credentials) >= 4) {
            Notification::make()
                ->title(__('Limit Reached'))
                ->warning()
                ->body(__('You can add a maximum of 4 credentials.'))
                ->send();
            return;
        }

        // Generate a UUID as the key for the new credential
        $uuid = (string) \Illuminate\Support\Str::uuid();

        // Add a new credential with the UUID key
        $this->credentials[$uuid] = [
            'credential_type' => \App\Enums\ProfessionalCredentionals::CertifiedBy->value,
            'credential' => ''
        ];

        $this->isDirty = true;
    }

    public function removeCredential($key)
    {
        // Directly remove the credential with the specific key
        unset($this->credentials[$key]);

        // Make sure we're working with the actual credentials array, not references
        $this->credentials = array_filter($this->credentials);

        $this->isDirty = true;
    }

    public function addAward()
    {
        if (count($this->awards_recognitions) >= 4) {
            Notification::make()
                ->title(__('Limit Reached'))
                ->warning()
                ->body(__('You can add a maximum of 4 awards.'))
                ->send();
            return;
        }

        // Generate a UUID as the key for the new award
        $uuid = (string) \Illuminate\Support\Str::uuid();

        // Add a new award with the UUID key
        $this->awards_recognitions[$uuid] = [
            'title' => '',
            'year' => date('Y')
        ];

        $this->isDirty = true;
    }

    public function removeAward($key)
    {
        // Directly remove the award with the specific key
        unset($this->awards_recognitions[$key]);

        // Make sure we're working with the actual awards array, not references
        $this->awards_recognitions = array_filter($this->awards_recognitions);

        $this->isDirty = true;
    }

    public function rules()
    {
        return [
            'first_name' => ['required', 'string', 'max:255'],
            'last_name' => ['required', 'string', 'max:255'],
            'company_name' => ['nullable', 'string', 'max:255'],
            'website' => ['nullable', 'url', 'max:255'],
            'email' => ['required', 'email', Rule::unique('users', 'email')->ignore($this->user->id)],
            'phone' => ['required', 'string', 'max:20'],
            'mobile' => ['nullable', 'string', 'max:20'],
            'year_established' => ['nullable', 'integer', 'min:1800', 'max:' . date('Y')],
            'slogan' => ['nullable', 'string', 'max:200'],
            'about' => ['nullable', 'string'],
            'avatar' => ['nullable', 'image', 'max:5120'], // Max 5MB
            'facebook' => ['nullable', 'url', 'max:255'],
            'instagram' => ['nullable', 'url', 'max:255'],
            'twitter' => ['nullable', 'url', 'max:255'],
            'linkedin' => ['nullable', 'url', 'max:255'],
            'youtube' => ['nullable', 'url', 'max:255'],
            'tiktok' => ['nullable', 'url', 'max:255'],
            'pinterest' => ['nullable', 'url', 'max:255'],
            'business_hours' => ['nullable', 'array'],
            'business_hours.*.day_of_week' => ['required', 'string'],
            'business_hours.*.is_open' => ['boolean'],
            'business_hours.*.hours_label' => ['nullable', 'string', 'max:30'],
            'business_hours.*.open_time' => [
                'nullable',
                function ($attribute, $value, $fail) {
                    // Extract the day key from the attribute (e.g., business_hours.monday.open_time -> monday)
                    preg_match('/business_hours\.(.+)\.open_time/', $attribute, $matches);
                    $day = $matches[1] ?? null;

                    if ($day && isset($this->business_hours[$day]['is_open']) && $this->business_hours[$day]['is_open']) {
                        if (empty($value)) {
                            $fail(__('Open time is required when the day is marked as open.'));
                        } elseif (!preg_match('/^([01]?[0-9]|2[0-3]):[0-5][0-9]$/', $value)) {
                            $fail(__('Invalid open time format.'));
                        }
                    }
                }
            ],
            'business_hours.*.close_time' => [
                'nullable',
                function ($attribute, $value, $fail) {
                    // Extract the day key from the attribute
                    preg_match('/business_hours\.(.+)\.close_time/', $attribute, $matches);
                    $day = $matches[1] ?? null;

                    if ($day && isset($this->business_hours[$day]['is_open']) && $this->business_hours[$day]['is_open']) {
                        if (empty($value)) {
                            $fail(__('Close time is required when the day is marked as open.'));
                        } elseif (!preg_match('/^([01]?[0-9]|2[0-3]):[0-5][0-9]$/', $value)) {
                            $fail(__('Invalid close time format.'));
                        } elseif (isset($this->business_hours[$day]['open_time']) && $this->business_hours[$day]['open_time'] && $value <= $this->business_hours[$day]['open_time']) {
                            $fail(__('Close time must be after open time.'));
                        }
                    }
                }
            ],
            'seasonal_hours' => ['nullable', 'array', 'max:7'],
            'seasonal_hours.*.is_open' => ['boolean'],
            'seasonal_hours.*.season_name' => ['required', 'string', 'max:20'],
            'seasonal_hours.*.start_date' => ['required', 'date'],
            'seasonal_hours.*.end_date' => ['required', 'date', 'after_or_equal:seasonal_hours.*.start_date'],
            'seasonal_hours.*.days' => ['required', 'string', 'max:20'],
            'seasonal_hours.*.start_time' => ['required', 'date_format:H:i'],
            'seasonal_hours.*.end_time' => ['required', 'date_format:H:i', 'after:seasonal_hours.*.start_time'],
            'profile_tags' => ['nullable', 'array', 'max:3'],
            'profile_tags.*' => ['string', 'max:20'],
            'credentials' => ['nullable', 'array'],
            'credentials.*.credential_type' => ['required', 'string', Rule::in(array_map(fn($case) => $case->value, \App\Enums\ProfessionalCredentionals::cases()))],
            'credentials.*.credential' => ['required', 'string', 'max:100'],
            'awards_recognitions' => ['nullable', 'array'],
            'awards_recognitions.*.title' => ['required', 'string', 'max:100'],
            'awards_recognitions.*.year' => ['required', 'integer', 'min:1800', 'max:' . date('Y')],
            'business_timezone' => ['nullable', 'string', 'timezone'],
        ];
    }

    public function messages()
    {
        return [
            'first_name.required' => __('The first name field is required.'),
            'last_name.required' => __('The last name field is required.'),
            'company_name.required' => __('The business name field is required.'),
            'website.url' => __('Please enter a valid website URL.'),
            'email.required' => __('The email field is required.'),
            'email.email' => __('Please enter a valid email address.'),
            'email.unique' => __('This email is already taken.'),
            'phone.required' => __('The contact number field is required.'),
            'mobile.max' => __('The mobile number cannot exceed 20 characters.'),
            'year_established.integer' => __('Year established must be a valid year.'),
            'year_established.min' => __('Year established must be after 1800.'),
            'year_established.max' => __('Year established cannot be in the future.'),
            'slogan.max' => __('The business slogan cannot exceed 200 characters.'),
            'about.required' => __('The business description field is required.'),
            'about.max' => __('The business description cannot exceed 200 characters.'),
            'avatar.image' => __('The file must be an image.'),
            'avatar.max' => __('The image must not be larger than 1MB.'),
            'facebook.url' => __('Please enter a valid Facebook URL.'),
            'instagram.url' => __('Please enter a valid Instagram URL.'),
            'twitter.url' => __('Please enter a valid X/Twitter URL.'),
            'linkedin.url' => __('Please enter a valid LinkedIn URL.'),
            'youtube.url' => __('Please enter a valid YouTube URL.'),
            'tiktok.url' => __('Please enter a valid TikTok URL.'),
            'pinterest.url' => __('Please enter a valid Pinterest URL.'),
            'business_hours.*.day_of_week.required' => __('Day of week is required.'),
            'seasonal_hours.*.season_name.required' => __('The season name is required.'),
            'seasonal_hours.*.start_date.required' => __('Start date is required.'),
            'seasonal_hours.*.start_date.date' => __('Please enter a valid date.'),
            'seasonal_hours.*.end_date.required' => __('End date is required.'),
            'seasonal_hours.*.end_date.date' => __('Please enter a valid date.'),
            'seasonal_hours.*.end_date.after_or_equal' => __('End date must be after or equal to start date.'),
            'seasonal_hours.*.days.required' => __('The days field is required.'),
            'seasonal_hours.*.start_time.required' => __('Start time is required.'),
            'seasonal_hours.*.start_time.date_format' => __('Invalid time format.'),
            'seasonal_hours.*.end_time.required' => __('End time is required.'),
            'seasonal_hours.*.end_time.date_format' => __('Invalid time format.'),
            'seasonal_hours.*.end_time.after' => __('End time must be after start time.'),
            'profile_tags.max' => __('You can add a maximum of 3 profile tags.'),
            'profile_tags.*.max' => __('Each profile tag cannot exceed 20 characters.'),
            'credentials.max' => __('You can add a maximum of 4 credentials.'),
            'credentials.*.credential_type.required' => __('Credential type is required.'),
            'credentials.*.credential_type.in' => __('Invalid credential type selected.'),
            'credentials.*.credential.required' => __('Credential field cannot be empty.'),
            'credentials.*.credential.max' => __('Credential cannot exceed 100 characters.'),
            'seasonal_hours.*.season_name.max' => __('Season name cannot exceed 20 characters.'),
            'seasonal_hours.*.days.max' => __('Days field cannot exceed 20 characters.'),
            'awards_recognitions.max' => __('You can add a maximum of 4 awards.'),
            'awards_recognitions.*.title.required' => __('Award title is required.'),
            'awards_recognitions.*.title.max' => __('Award title cannot exceed 100 characters.'),
            'awards_recognitions.*.year.required' => __('Award year is required.'),
            'awards_recognitions.*.year.integer' => __('Award year must be a valid year.'),
            'awards_recognitions.*.year.min' => __('Award year must be after 1800.'),
            'awards_recognitions.*.year.max' => __('Award year cannot be in the future.'),
            'business_timezone.timezone' => __('Please select a valid timezone.'),
        ];
    }

    public function updateProfile()
    {
        $validatedData = $this->validate();

        try {
            DB::beginTransaction();
            // Update user email if changed
            if ($this->user->email !== $validatedData['email']) {
                $this->user->update(['email' => $validatedData['email']]);
            }

            // Prepare social media links
            $this->social_media_links = array_filter([
                'facebook' => $validatedData['facebook'],
                'instagram' => $validatedData['instagram'],
                'twitter' => $validatedData['twitter'],
                'linkedin' => $validatedData['linkedin'],
                'youtube' => $validatedData['youtube'],
                'tiktok' => $validatedData['tiktok'],
                'pinterest' => $validatedData['pinterest'],
            ]);


            // Filter out empty credentials but maintain UUID keys
            $filteredCredentials = [];
            foreach ($this->credentials as $key => $credential) {
                if (!empty($credential['credential'])) {
                    $filteredCredentials[$key] = $credential;
                }
            }

            // Filter out empty awards but maintain UUID keys
            $filteredAwards = [];
            foreach ($this->awards_recognitions as $key => $award) {
                if (!empty($award['title'])) {
                    $filteredAwards[$key] = $award;
                }
            }

            // Update or create profile
            $profileData = [
                'first_name' => $validatedData['first_name'],
                'last_name' => $validatedData['last_name'],
                'company_name' => $validatedData['company_name'],
                'website' => $validatedData['website'],
                'phone' => $validatedData['phone'],
                'mobile' => $validatedData['mobile'],
                'year_established' => $validatedData['year_established'],
                'slogan' => $validatedData['slogan'],
                'about' => $validatedData['about'],
                'social_media_links' => $this->social_media_links,
                'credentials' => $filteredCredentials,
            ];

            if ($this->profile) {
                $this->profile->update($profileData);
            } else {
                $this->profile = $this->user->profile()->create($profileData);
            }

            // Save business hours using the service - always try to save if we have business hours data
            if ($this->business_hours) {
                $businessHourService = app(BusinessHourService::class);

                // Convert the business hours data to the format expected by the service
                $businessHoursData = [];
                foreach ($this->business_hours as $day => $dayData) {
                    $businessHoursData[] = [
                        'day_of_week' => $dayData['day_of_week'],
                        'is_open' => $dayData['is_open'] ?? false,
                        'hours_label' => $dayData['hours_label'] ?? null,
                        'open_time' => ($dayData['is_open'] ?? false) ? ($dayData['open_time'] ?? null) : null,
                        'close_time' => ($dayData['is_open'] ?? false) ? ($dayData['close_time'] ?? null) : null,
                    ];
                }
                $result = $businessHourService->createOrUpdateBusinessHours($this->user, $businessHoursData);
            }

            // Save meta data
            $this->profile->setMeta('awards_recognitions', $filteredAwards);
            $this->profile->setMeta('seasonal_hours', $this->seasonal_hours);
            $this->profile->setMeta('business_timezone', $this->business_timezone);
            // Filter and save profile tags (max 3, each <= 20 chars, unique)
            $profileTags = collect($this->profile_tags ?? [])
                ->filter(fn($t) => is_string($t) && trim($t) !== '')
                ->map(fn($t) => Str::limit(trim($t), 20, ''))
                ->unique()
                ->take(3)
                ->values()
                ->toArray();
            $this->profile->setMeta('profile_tags', $profileTags);
            $this->profile->save();

            // Handle avatar upload if provided
            if ($this->avatar) {
                $this->user->addMedia($this->avatar)
                    ->toMediaCollection('avatar');
            }

            DB::commit();
            $this->isDirty = false;

            Notification::make()
                ->title(__('Success!'))
                ->success()
                ->body(__('Your profile has been updated successfully.'))
                ->send();
        } catch (\Exception $e) {
            DB::rollBack();

            Notification::make()
                ->title(__('Error!'))
                ->danger()
                ->body(__('An error occurred while updating your profile. Please try again.' . $e->getMessage()))
                ->send();
        }
    }

    public function render()
    {
        return view('livewire.dashboard.profile.profile-basic');
    }
}
