<?php

namespace App\Livewire\Dashboard\Profile;

use Livewire\Component;
use Livewire\Attributes\Layout;
use App\Services\User\MemberCategoryService;
use Illuminate\Database\Eloquent\Collection;
use Livewire\Attributes\Validate;
use Illuminate\Support\Facades\Auth;
use App\Models\User\User;
use Filament\Notifications\Notification;

#[Layout('layouts.dashboard.dashboard-app')]
class ProfileBusinessMember extends Component
{
    public Collection $parentCategories;
    public ?string $selectedParentId = null;
    public Collection $selectedCategories;
    public array $childCategoriesMap = [];
    public array $selectedParentIds = [];
    public array $secondLevelCategoriesMap = [];
    public array $selectedChildIds = [];

    #[Validate('required|array|min:1', message: 'Please select at least one category')]
    public array $selectedCategoryIds = [];

    public User $user;

    public function mount(MemberCategoryService $categoryService)
    {
        $this->user = Auth::user();
        $this->parentCategories = $categoryService->getParentCategories();
        $this->selectedCategories = new Collection();

        // Initialize with empty collections - no saving required
         // If user has existing categories, load them
         if (Auth::check() && $this->user instanceof User) {
            $userCategories = $this->user->memberCategories()->with('parent')->get();
            if ($userCategories && $userCategories->isNotEmpty()) {
                $this->selectedCategories = $userCategories;
                $this->selectedCategoryIds = $userCategories->pluck('id')->toArray();

                // First, identify and load parent categories
                $parentCategories = $userCategories->whereNull('parent_id');
                foreach ($parentCategories as $parent) {
                    $this->selectedParentIds[] = $parent->id;
                    $this->loadChildCategories($parent->id);
                }

                // Then, process child categories only if their parent is selected
                foreach ($userCategories as $category) {
                    if ($category->parent_id && in_array($category->parent_id, $this->selectedParentIds)) {
                        $this->selectedChildIds[] = $category->id;
                        $this->loadSecondLevelCategories($category->id);
                    }
                }
            }
        }
    }
    protected function rules(): array
    {
        return [
            'selectedCategoryIds' => 'required|array|min:1',
        ];
    }
    
    protected function messages(): array
    {
        return [
            'selectedCategoryIds.required' => 'Please select at least one category.',
            'selectedCategoryIds.min' => 'Please select at least one category.',
        ];
    }

    public function updatedSelectedParentId($value)
    {
        if (!$value) {
            return;
        }

        $category = app(MemberCategoryService::class)->findById($value);

        if ($category && !in_array($category->id, $this->selectedParentIds)) {
            $this->selectedCategories->push($category);
            $this->selectedCategoryIds[] = $category->id;
            $this->selectedParentIds[] = $category->id;
            $this->loadChildCategories($category->id);
        }

        $this->selectedParentId = null; // Reset dropdown
    }

    protected function loadChildCategories($parentId)
    {
        $children = app(MemberCategoryService::class)->getChildCategories($parentId);
        if ($children->isNotEmpty()) {
            $this->childCategoriesMap[$parentId] = $children;
        }
    }

    protected function loadSecondLevelCategories($childId)
    {
        $secondLevel = app(MemberCategoryService::class)->getChildCategories($childId);
        if ($secondLevel->isNotEmpty()) {
            $this->secondLevelCategoriesMap[$childId] = $secondLevel;
        }
    }

    public function toggleChildCategory($categoryId)
    {
        if (in_array($categoryId, $this->selectedCategoryIds)) {
            // Remove category and its children
            $this->selectedCategories = $this->selectedCategories->filter(
                fn($category) =>
                $category->id != $categoryId &&
                    (!isset($this->secondLevelCategoriesMap[$categoryId]) ||
                        !collect($this->secondLevelCategoriesMap[$categoryId])->contains('id', $category->id))
            )->values();

            $this->selectedCategoryIds = array_values(array_filter(
                $this->selectedCategoryIds,
                fn($id) =>
                $id != $categoryId &&
                    (!isset($this->secondLevelCategoriesMap[$categoryId]) ||
                        !in_array($id, collect($this->secondLevelCategoriesMap[$categoryId])->pluck('id')->toArray()))
            ));

            $this->selectedChildIds = array_values(array_diff($this->selectedChildIds, [$categoryId]));
            unset($this->secondLevelCategoriesMap[$categoryId]);
        } else {
            // Add category
            $category = app(MemberCategoryService::class)->findById($categoryId);
            if ($category) {
                $this->selectedCategories->push($category);
                $this->selectedCategoryIds[] = $categoryId;
                $this->selectedChildIds[] = $categoryId;
                $this->loadSecondLevelCategories($categoryId);
            }
        }
    }

    public function toggleSecondLevelCategory($categoryId)
    {
        if (in_array($categoryId, $this->selectedCategoryIds)) {
            // Remove category
            $this->selectedCategories = $this->selectedCategories->filter(fn($category) => $category->id != $categoryId)->values();
            $this->selectedCategoryIds = array_values(array_filter($this->selectedCategoryIds, fn($id) => $id != $categoryId));
        } else {
            // Add category
            $category = app(MemberCategoryService::class)->findById($categoryId);
            if ($category) {
                $this->selectedCategories->push($category);
                $this->selectedCategoryIds[] = $categoryId;
            }
        }
    }

    public function removeCategory($categoryId)
    {
        // If it's a parent category, remove all its children and second-level children first
        if (in_array($categoryId, $this->selectedParentIds)) {
            // Get all child categories for this parent
            if (isset($this->childCategoriesMap[$categoryId])) {
                foreach ($this->childCategoriesMap[$categoryId] as $child) {
                    // Remove second-level categories for each child
                    if (isset($this->secondLevelCategoriesMap[$child->id])) {
                        foreach ($this->secondLevelCategoriesMap[$child->id] as $secondLevel) {
                            $this->selectedCategories = $this->selectedCategories->filter(
                                fn($category) =>
                                $category->id != $secondLevel->id
                            );
                            $this->selectedCategoryIds = array_values(array_filter(
                                $this->selectedCategoryIds,
                                fn($id) =>
                                $id != $secondLevel->id
                            ));
                        }
                        unset($this->secondLevelCategoriesMap[$child->id]);
                    }

                    // Remove child category
                    $this->selectedCategories = $this->selectedCategories->filter(
                        fn($category) =>
                        $category->id != $child->id
                    );
                    $this->selectedCategoryIds = array_values(array_filter(
                        $this->selectedCategoryIds,
                        fn($id) =>
                        $id != $child->id
                    ));
                    $this->selectedChildIds = array_values(array_diff($this->selectedChildIds, [$child->id]));
                }
                unset($this->childCategoriesMap[$categoryId]);
            }

            // Remove parent category
            $this->selectedParentIds = array_values(array_diff($this->selectedParentIds, [$categoryId]));
        }
        // If it's a child category, remove its second-level children
        elseif (in_array($categoryId, $this->selectedChildIds)) {
            if (isset($this->secondLevelCategoriesMap[$categoryId])) {
                foreach ($this->secondLevelCategoriesMap[$categoryId] as $secondLevel) {
                    $this->selectedCategories = $this->selectedCategories->filter(
                        fn($category) =>
                        $category->id != $secondLevel->id
                    );
                    $this->selectedCategoryIds = array_values(array_filter(
                        $this->selectedCategoryIds,
                        fn($id) =>
                        $id != $secondLevel->id
                    ));
                }
                unset($this->secondLevelCategoriesMap[$categoryId]);
            }
            $this->selectedChildIds = array_values(array_diff($this->selectedChildIds, [$categoryId]));
        }

        // Remove the category itself
        $this->selectedCategories = $this->selectedCategories->filter(
            fn($category) =>
            $category->id != $categoryId
        )->values();

        $this->selectedCategoryIds = array_values(array_filter(
            $this->selectedCategoryIds,
            fn($id) =>
            $id != $categoryId
        ));
    }

    public function saveCategories()
    {
        $this->validate();

        try {
            if (Auth::check() && $this->user instanceof User) {
                // Clear existing relationships first
                $this->user->memberCategories()->detach();

                // Then sync new relationships
                if (!empty($this->selectedCategoryIds)) {
                    $this->user->memberCategories()->sync($this->selectedCategoryIds);
                } else {
                    $this->user->memberCategories()->sync([]);
                }

                Notification::make()
                    ->title('Categories saved successfully')
                    ->success()
                    ->send();
            } else {
                Notification::make()
                    ->title('Authentication error')
                    ->body('User not authenticated or invalid user type.')
                    ->danger()
                    ->send();
            }
        } catch (\Exception $e) {
            Notification::make()
                ->title('Error saving categories')
                ->body('Failed to update categories. Please try again.')
                ->danger()
                ->send();
        }
    }
    public function render()
    {
        return view('livewire.dashboard.profile.profile-business-member', [
            'selectedCategories' => $this->selectedCategories,
            'childCategoriesMap' => $this->childCategoriesMap,
            'selectedParentIds' => $this->selectedParentIds,
            'secondLevelCategoriesMap' => $this->secondLevelCategoriesMap,
            'selectedChildIds' => $this->selectedChildIds
        ]);
    }
}
