<?php

declare(strict_types=1);

namespace App\Livewire\Dashboard\Profile;

use Filament\Notifications\Notification;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Livewire\Attributes\Layout;
use Livewire\Attributes\Rule;
use Livewire\Component;
use Livewire\WithFileUploads;

#[Layout('layouts.dashboard.dashboard-app')]
class ProfileGallaryImages extends Component
{
    use WithFileUploads;

    public $user;

    #[Rule('array|min:1|max:10')]
    public $images = [];

    #[Rule('array')] // Allow multiple images
    public $newImages = [];

    public $urlImage = '';

    public function mount()
    {
        $this->user = Auth::user();
        $this->loadGalleryImages();
    }

    public function loadGalleryImages()
    {
        $this->images = $this->user->getMedia('gallery')
            ->sortBy('order_column')
            ->map(function ($media) {
                return [
                    'id' => $media->id,
                    'url' => $media->getUrl('thumb'),
                    'name' => $media->name,
                    'order' => $media->order_column,
                ];
            })->values()->toArray();
    }

    public function updateImageOrder($items)
    {
        try {
            // Begin transaction
            DB::beginTransaction();

            $gallery = $this->user->getMedia('gallery');

            // Create a mapping of current order positions
            $orderMap = [];

            // Update order for each image
            foreach ($items as $index => $item) {
                $mediaId = $item['value'];

                // Skip temporary IDs
                if (is_string($mediaId) && strpos($mediaId, 'temp_') === 0) {
                    // Track order for temp items
                    $orderMap[$mediaId] = $index + 1;
                    continue;
                }

                $media = $gallery->where('id', $mediaId)->first();

                if ($media) {
                    $media->update(['order_column' => $index + 1]);
                    $orderMap[$mediaId] = $index + 1;
                }
            }

            // Update order for temporary images in the images array
            foreach ($this->images as $key => $image) {
                if (isset($image['id']) && isset($orderMap[$image['id']])) {
                    $this->images[$key]['order'] = $orderMap[$image['id']];
                }
            }

            // Commit transaction
            DB::commit();

            // Re-sort the images array by order
            $this->images = collect($this->images)->sortBy('order')->values()->toArray();

            Notification::make()
                ->success()
                ->title(__('Order updated successfully'))
                ->send();
        } catch (\Exception $e) {
            // Rollback transaction
            DB::rollBack();

            Notification::make()
                ->danger()
                ->title(__('Failed to update order'))
                ->send();

            Log::error('Failed to update gallery image order: ' . $e->getMessage());
        }
    }

    public function updatedNewImages()
    {
        try {
            // Check total image count (existing + new)
            $totalImages = count($this->images) + count($this->newImages);
            
            if ($totalImages > 10) {
                $this->newImages = [];
                Notification::make()
                    ->danger()
                    ->title(__('Maximum image limit reached'))
                    ->body(__('You can only upload up to 10 images in total.'))
                    ->send();
                return;
            }

            $this->validate([
                'newImages.*' => 'image|max:5120',
            ], [], [
                'newImages.*' => 'Image',
            ]);

            $maxOrder = collect($this->images)->max('order') ?? 0;

            foreach ($this->newImages as $index => $image) {
                try {
                    // Get the image data as base64
                    $imageData = base64_encode(file_get_contents($image->getRealPath()));
                    $imageType = $image->getMimeType();
                    $previewUrl = "data:{$imageType};base64,{$imageData}";

                    $this->images[] = [
                        'id' => 'temp_' . uniqid(),
                        'url' => $previewUrl,
                        'name' => $image->getClientOriginalName(),
                        'temp' => true,
                        'file' => $image,
                        'order' => $maxOrder + $index + 1,
                    ];
                } catch (\Exception $e) {
                    Log::error('Failed to process image: ' . $e->getMessage());
                    Notification::make()
                        ->danger()
                        ->title(__('Failed to preview image'))
                        ->send();

                    continue;
                }
            }

            $this->newImages = [];
        } catch (\Illuminate\Validation\ValidationException $e) {
            foreach ($e->errors() as $field => $errors) {
                Notification::make()
                    ->danger()
                    ->title($errors[0])
                    ->send();
            }
            $this->newImages = [];
            throw $e;
        }
    }

    public function addImageFromUrl()
    {
        try {
            $this->validate([
                'urlImage' => 'required|url',
            ]);

            $tempImage = file_get_contents($this->urlImage);
            $tempPath = tempnam(sys_get_temp_dir(), 'url_image');
            file_put_contents($tempPath, $tempImage);

            $this->images[] = [
                'id' => null,
                'url' => $this->urlImage,
                'name' => basename($this->urlImage),
                'temp' => true,
                'path' => $tempPath,
            ];

            $this->urlImage = '';

            Notification::make()
                ->success()
                ->title(__('Image from URL added successfully'))
                ->send();
        } catch (\Illuminate\Validation\ValidationException $e) {
            Notification::make()
                ->danger()
                ->title(__('Invalid image URL'))
                ->send();
            throw $e;
        } catch (\Exception $e) {
            Log::error('Failed to add image from URL: ' . $e->getMessage());
            Notification::make()
                ->danger()
                ->title(__('Failed to load image'))
                ->body(__('Please check the URL and try again.'))
                ->send();
        }
    }

    public function deleteImage($index)
    {
        try {
            if (isset($this->images[$index]['id']) && !is_string($this->images[$index]['id']) || (strpos($this->images[$index]['id'], 'temp_') !== 0)) {
                $media = $this->user->getMedia('gallery')->where('id', $this->images[$index]['id'])->first();
                if ($media) {
                    $media->delete();
                    Notification::make()
                        ->success()
                        ->title(__('Image deleted successfully'))
                        ->send();
                }
            } else {
                // For temporary images, just remove from array
                Notification::make()
                    ->success()
                    ->title(__('Image removed'))
                    ->send();
            }

            unset($this->images[$index]);
            $this->images = array_values($this->images);
        } catch (\Exception $e) {
            Log::error('Failed to delete image: ' . $e->getMessage());
            Notification::make()
                ->danger()
                ->title(__('Failed to delete image'))
                ->body(__('Please try again.'))
                ->send();
        }
    }

    public function save()
    {
        try {
            $this->validate([
                'images' => 'array|min:1|max:10',
            ], [], [
                'images' => 'Gallery Images',
            ]);

            $savedCount = 0;

            foreach ($this->images as $index => $image) {
                // Check if this is a new image (either a string ID starting with 'temp_' or null ID)
                if (!isset($image['id']) || (is_string($image['id']) && strpos($image['id'], 'temp_') === 0)) {
                    if (isset($image['file'])) {
                        // Handle uploaded files
                        $this->user->addMedia($image['file']->getRealPath())
                            ->withCustomProperties(['order_column' => $index])
                            ->usingName($image['name'])
                            ->toMediaCollection('gallery');
                        $savedCount++;
                    } elseif (isset($image['path'])) {
                        // Handle URL images
                        $this->user->addMedia($image['path'])
                            ->withCustomProperties(['order_column' => $index])
                            ->usingName($image['name'])
                            ->toMediaCollection('gallery');
                        $savedCount++;
                    }
                }
            }

            $this->dispatch('gallery-updated');

            if ($savedCount > 0) {
                Notification::make()
                    ->success()
                    ->title(__('Gallery images updated successfully'))
                    ->send();
            } else {
                Notification::make()
                    ->info()
                    ->title(__('No new images to save.'))
                    ->send();
            }

            $this->loadGalleryImages();
        } catch (\Illuminate\Validation\ValidationException $e) {
            foreach ($e->errors() as $field => $errors) {
                Notification::make()
                    ->danger()
                    ->title($errors[0])
                    ->send();
            }
            throw $e;
        } catch (\Exception $e) {
            Log::error('Failed to save gallery images: ' . $e->getMessage());
            Notification::make()
                ->danger()
                ->title(__('Failed to save gallery images'))
                ->body(__('Please try again.'))
                ->send();
        }
    }

    public function render()
    {
        return view('livewire.dashboard.profile.profile-gallary-images');
    }
}
