<?php

declare(strict_types=1);

namespace App\Livewire\Dashboard\Profile;

use App\Services\User\UserLocationService;
use Illuminate\Support\Facades\Auth;
use Livewire\Component;
use Livewire\Attributes\Layout;
use Filament\Actions\Action;
use Filament\Actions\Concerns\InteractsWithActions;
use Filament\Actions\Contracts\HasActions;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Notifications\Notification;

#[Layout('layouts.dashboard.dashboard-app')]
class ProfileLocation extends Component implements HasForms, HasActions
{
    use InteractsWithForms;
    use InteractsWithActions;

    protected UserLocationService $locationService;

    public function boot(UserLocationService $locationService): void
    {
        $this->locationService = $locationService;
    }

    public function deleteAction(): Action
    {
        return Action::make('delete')
            ->requiresConfirmation()
            ->color('danger')
            ->button()
            ->label('Remove')
            ->modalIcon('heroicon-o-trash')
            ->modalIconColor('danger')
            ->modalHeading('Delete Location')
            ->modalDescription('Are you sure you want to delete this location? This action cannot be undone.')
            ->modalSubmitActionLabel('Yes, delete it')
            ->modalCancelActionLabel('No, cancel')
            ->mountUsing(fn(Action $action, array $arguments) => $action->arguments(['uuid' => $arguments['uuid']]))
            ->action(function (Action $action) {
                try {
                    $arguments = $action->getArguments();
                    if (isset($arguments['uuid'])) {
                        $location = $this->locationService->getLocationByUuid($arguments['uuid']);
                        
                        if ($location) {
                            $this->locationService->deleteLocation($location);
                            
                            Notification::make()
                                ->title('Location deleted successfully')
                                ->success()
                                ->send();
                        }
                    }
                } catch (\Exception $e) {
                    Notification::make()
                        ->title('Error')
                        ->body('There was an error deleting the location.'. $e->getMessage())
                        ->danger()
                        ->send();
                }
            });
    }

    public function render()
    {
        $locations = $this->locationService->getActiveLocations(Auth::id());
        
        return view('livewire.dashboard.profile.profile-location', [
            'locations' => $locations,
        ]);
    }
}
