<?php

declare(strict_types=1);

namespace App\Livewire\Dashboard\Profile;

use App\Models\User\UserLocation;
use App\Services\User\UserLocationService;
use App\Services\Location\AreaService;
use App\Services\Location\CountryService;
use App\Services\Location\StateService;
use App\Services\Location\CityService;
use Illuminate\Support\Facades\Auth;
use Livewire\Component;
use Livewire\Attributes\Rule;
use Livewire\Attributes\Layout;
use Illuminate\Support\Collection;
use Filament\Notifications\Notification;
use Filament\Notifications\Actions\Action;

#[Layout('layouts.dashboard.dashboard-app')]
class ProfileLocationAddEdit extends Component
{
    #[Rule('required|string|max:255')]
    public string $location_name = '';

    #[Rule('nullable|string|max:255')]
    public ?string $contact_person_name = '';

    #[Rule('required|string|max:20')]
    public string $phone = '';

    #[Rule('required|email|max:255')]
    public string $email = '';

    #[Rule('nullable|string|max:20')]
    public ?string $fax = '';

    #[Rule('required|string|max:255')]
    public string $address_line_1 = '';

    #[Rule('nullable|string|max:255')]
    public ?string $address_line_2 = '';

    #[Rule('nullable|string|max:20')]
    public ?string $postal_code = '';

    #[Rule('required|string|max:255')]
    public string $map_address = '';

    #[Rule('required|string|max:255')]
    public string $country_name = '';

    #[Rule('required|string|max:10')]
    public ?string $country_code = '';

    #[Rule('required|string|max:255')]
    public string $state_name = '';

    #[Rule('nullable|string|max:10')]
    public ?string $state_code = '';

    #[Rule('required|string|max:255')]
    public string $city_name = '';

    public ?float $latitude = null;
    public ?float $longitude = null;

    #[Rule('nullable|array')]
    public ?array $service_areas = [];

    public bool $primary_location = false;
    public bool $is_active = true;

    public ?string $locationUuid = null;
    public bool $isEditing = false;

    private array $originalData = [];
    public bool $isDirty = false;

    protected UserLocationService $locationService;
    protected AreaService $areaService;
    protected CountryService $countryService;
    protected StateService $stateService;
    protected CityService $cityService;

    protected $listeners = [
        'placeSelected' => 'handlePlaceSelected'
    ];

    public function boot()
    {
        $this->locationService = app(UserLocationService::class);
        $this->areaService = app(AreaService::class);
        $this->countryService = app(CountryService::class);
        $this->stateService = app(StateService::class);
        $this->cityService = app(CityService::class);
    }

    public function mount(?string $uuid = null)
    {
        if ($uuid) {
            $this->locationUuid = $uuid;
            $this->isEditing = true;
            $this->loadLocation();
        }

        // Store original data after loading
        $this->storeOriginalData();
    }

    private function storeOriginalData(): void
    {
        $this->originalData = [
            'location_name' => $this->location_name,
            'contact_person_name' => $this->contact_person_name,
            'phone' => $this->phone,
            'email' => $this->email,
            'fax' => $this->fax,
            'address_line_1' => $this->address_line_1,
            'address_line_2' => $this->address_line_2,
            'postal_code' => $this->postal_code,
            'map_address' => $this->map_address,
            'country_name' => $this->country_name,
            'country_code' => $this->country_code,
            'state_name' => $this->state_name,
            'state_code' => $this->state_code,
            'city_name' => $this->city_name,
            'primary_location' => $this->primary_location,
            'latitude' => $this->latitude,
            'longitude' => $this->longitude,
            'service_areas' => $this->service_areas,
        ];
    }

    public function updated($field): void
    {
        if ($field !== 'isDirty') {
            $this->checkIfDirty();
            $this->validateOnly($field);
        }
    }

    private function checkIfDirty(): void
    {
        $currentData = [
            'location_name' => $this->location_name,
            'contact_person_name' => $this->contact_person_name,
            'phone' => $this->phone,
            'email' => $this->email,
            'fax' => $this->fax,
            'address_line_1' => $this->address_line_1,
            'address_line_2' => $this->address_line_2,
            'postal_code' => $this->postal_code,
            'map_address' => $this->map_address,
            'country_name' => $this->country_name,
            'country_code' => $this->country_code,
            'state_name' => $this->state_name,
            'state_code' => $this->state_code,
            'city_name' => $this->city_name,
            'primary_location' => $this->primary_location,
            'latitude' => $this->latitude,
            'longitude' => $this->longitude,
            'service_areas' => $this->service_areas,
        ];

        $this->isDirty = $this->originalData != $currentData;
    }

    public function loadLocation(): void
    {
        if (!$this->locationUuid) {
            return;
        }

        $location = $this->locationService->getLocationByUuid($this->locationUuid);

        if ($location) {
            $this->location_name = $location->location_name;
            $this->contact_person_name = $location->contact_person_name ?? '';
            $this->phone = $location->phone;
            $this->email = $location->email;
            $this->fax = $location->fax ?? '';
            $this->address_line_1 = $location->address_line_1;
            $this->address_line_2 = $location->address_line_2 ?? '';
            $this->postal_code = $location->postal_code ?? '';
            $this->map_address = $location->map_address ?? '';
            $this->primary_location = $location->primary_location;
            $this->is_active = $location->is_active;
            $this->latitude = $location->latitude;
            $this->longitude = $location->longitude;

            // Initialize with empty strings
            $this->country_name = '';
            $this->country_code = '';
            $this->state_name = '';
            $this->state_code = '';
            $this->city_name = '';

            // Load location details
            if ($location->country) {
                $this->country_name = $location->country->country_name;
                $this->country_code = $location->country->country_code ?? '';
            }
            if ($location->state) {
                $this->state_name = $location->state->state_name;
                $this->state_code = $location->state->state_code ?? '';
            }
            if ($location->city) {
                $this->city_name = $location->city->city_name;
            }

            // Load service areas
            $this->service_areas = $location->areas->pluck('name')->toArray();
        }
    }

    public function handlePlaceSelected($placeData)
    {
        // Set map address and coordinates
        $this->map_address = $placeData['formatted_address'];
        $this->latitude = $placeData['latitude'];
        $this->longitude = $placeData['longitude'];

        $this->isDirty = true;

        // Initialize with empty strings
        $this->country_name = '';
        $this->country_code = '';
        $this->state_name = '';
        $this->state_code = '';
        $this->city_name = '';
        $this->postal_code = '';

        // Store the current address_line_1 value
        $currentAddressLine1 = $this->address_line_1;

        // Set address components
        foreach ($placeData['address_components'] as $component) {
            $types = $component['types'];

            if (in_array('country', $types)) {
                $this->country_name = $component['long_name'];
                $this->country_code = $component['short_name'];
            }
            if (in_array('administrative_area_level_1', $types)) {
                $this->state_name = $component['long_name'];
                $this->state_code = strlen($component['short_name']) > 10 ? null : $component['short_name'];
            }
            if (in_array('locality', $types)) {
                $this->city_name = $component['long_name'];
            }
            if (in_array('postal_code', $types)) {
                $this->postal_code = $component['long_name'];
            }

            // Only set address_line_1 if it's empty or hasn't been manually edited
            if (empty($currentAddressLine1)) {
                if (in_array('street_number', $types)) {
                    $this->address_line_1 = $component['long_name'];
                }
                if (in_array('route', $types)) {
                    $this->address_line_1 = ($this->address_line_1 ? $this->address_line_1 . ' ' : '') . $component['long_name'];
                }
            }
        }

        // If address_line_1 was manually set, keep it
        if (!empty($currentAddressLine1)) {
            $this->address_line_1 = $currentAddressLine1;
        }
    }

    public function addServiceArea($area)
    {
        $area = trim($area);
        if (!empty($area) && !in_array($area, $this->service_areas ?? [])) {
            $this->service_areas[] = $area;
            $this->isDirty = true;
        }
    }

    public function removeServiceArea($index)
    {
        if (isset($this->service_areas[$index])) {
            unset($this->service_areas[$index]);
            $this->service_areas = array_values($this->service_areas);
            $this->isDirty = true;
        }
    }

    public function save()
    {
        try {
            // Validate the input
            $validated = $this->validate();

            // Add latitude and longitude to location data
            $data = [
                'location_name' => $this->location_name,
                'contact_person_name' => $this->contact_person_name,
                'phone' => $this->phone,
                'email' => $this->email,
                'fax' => $this->fax,
                'address_line_1' => $this->address_line_1,
                'address_line_2' => $this->address_line_2,
                'postal_code' => $this->postal_code,
                'map_address' => $this->map_address,
                'latitude' => $this->latitude,
                'longitude' => $this->longitude,
                'primary_location' => $this->primary_location,
                'is_active' => $this->is_active,
            ];

            // Handle Country using service
            $country = $this->countryService->findOrCreateByCodeAndName(
                $this->country_code,
                $this->country_name
            );

            // Handle State using service
            $state = $this->stateService->findOrCreateByNameAndCountry(
                $this->state_name,
                $country->id,
                $this->state_code
            );

            // Handle City using service
            $city = $this->cityService->findOrCreateByNameAndState(
                $this->city_name,
                $state->id,
                $country->id
            );
            
            // Add location relationships
            $data['country_id'] = $country->id;
            $data['state_id'] = $state->id;
            $data['city_id'] = $city->id;

            if ($this->locationUuid) {
                $location = $this->locationService->getLocationByUuid($this->locationUuid);
                if ($location) {
                    // If setting this location as primary, remove primary from other locations
                    if ($this->primary_location) {
                        $this->locationService->removeOtherPrimaryLocations(Auth::id(), $location->id);
                    }

                    $location = $this->locationService->updateLocation($location, $data);

                    // Handle service areas
                    if (!empty($this->service_areas) && $city->id) {
                        $this->areaService->syncAreasToLocation($location, $this->service_areas);
                    } else {
                        $location->areas()->detach();
                    }

                    Notification::make()
                        ->success()
                        ->title('Location Updated')
                        ->body('Location has been updated successfully.')
                        ->send();
                }
            } else {
                $data['user_id'] = Auth::id();
                $data['created_by'] = Auth::id();

                // If setting this location as primary, remove primary from other locations
                if ($this->primary_location) {
                    $this->locationService->removeOtherPrimaryLocations(Auth::id());
                }

                $location = $this->locationService->createLocation($data);

                // Handle service areas
                if (!empty($this->service_areas) && $city->id) {
                    $this->areaService->syncAreasToLocation($location, $this->service_areas);
                }

                Notification::make()
                    ->success()
                    ->title('Location Created')
                    ->body('Location has been created successfully.')
                    ->send();
            }

            $this->isDirty = false;

            // Redirect after successful save
            $this->redirect(route('dashboard.profile.location.index'));
        } catch (\Illuminate\Validation\ValidationException $e) {
            $errors = collect($e->errors())->flatten()->implode("\n");

            Notification::make()
                ->danger()
                ->title('Validation Error')
                ->body($errors)
                ->persistent()
                ->send();

            throw $e;
        } catch (\Exception $e) {
            Notification::make()
                ->danger()
                ->title('Error')
                ->body('There was an error saving the location.')
                ->persistent()
                ->send();
        }
    }

    public function render()
    {
        return view('livewire.dashboard.profile.profile-location-add-edit');
    }
}
