<?php

namespace App\Livewire\Dashboard\Profile;

use Filament\Notifications\Notification;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Livewire\Attributes\Layout;
use Livewire\Attributes\Validate;
use Livewire\Component;
use Livewire\WithFileUploads;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use App\Models\User\User;

#[Layout('layouts.dashboard.dashboard-app')]
class ProfilePhoto extends Component
{
    use WithFileUploads;

    #[Validate('nullable|image|mimes:jpeg,png,jpg,webp|max:5120')]
    public $logo = null;

    #[Validate('nullable|image|mimes:jpeg,png,jpg,webp|max:5120')]
    public $cover = null;

    public $logoUrl = '';

    public $coverUrl = '';

    public $existingLogoUrl = null;

    public $existingCoverUrl = null;

    public $tempLogoUrl = null;

    public $tempCoverUrl = null;

    public $user;

    public function mount()
    {
        $this->user = Auth::user();
        if($this->user->getMedia('logo')->count() > 0){
            $this->existingLogoUrl = $this->user->getFirstMediaUrl('logo', 'medium');
        }
        if($this->user->getMedia('cover')->count() > 0){
            $this->existingCoverUrl = $this->user->getFirstMediaUrl('cover', 'medium');
        }
    }

    public function updatedLogo()
    {
        try {
            if ($this->logo) {
                $this->validate([
                    'logo' => 'image|mimes:jpeg,png,jpg,webp|max:5120',
                ]);

                // Generate base64 preview
                $imageData = base64_encode(file_get_contents($this->logo->getRealPath()));
                $imageType = $this->logo->getMimeType();
                $this->tempLogoUrl = "data:{$imageType};base64,{$imageData}";
            }
        } catch (\Exception $e) {
            Log::error('Failed to process logo: ' . $e->getMessage());
            Notification::make()
                ->danger()
                ->title(__('Failed to preview logo'))
                ->send();
            $this->reset(['logo', 'tempLogoUrl']);
        }
    }

    public function updatedCover()
    {
        try {
            if ($this->cover) {
                $this->validate([
                    'cover' => 'image|mimes:jpeg,png,jpg,webp|max:5120',
                ]);

                // Generate base64 preview
                $imageData = base64_encode(file_get_contents($this->cover->getRealPath()));
                $imageType = $this->cover->getMimeType();
                $this->tempCoverUrl = "data:{$imageType};base64,{$imageData}";
            }
        } catch (\Exception $e) {
            Log::error('Failed to process cover: ' . $e->getMessage());
            Notification::make()
                ->danger()
                ->title(__('Failed to preview cover image'))
                ->send();
            $this->reset(['cover', 'tempCoverUrl']);
        }
    }

    public function deleteLogo()
    {
        try {
            $this->user->clearMediaCollection('logo');
            $this->existingLogoUrl = null;
            $this->reset(['logo', 'logoUrl', 'tempLogoUrl']);

            Notification::make()
                ->title(__('Logo deleted successfully'))
                ->success()
                ->send();
        } catch (\Exception $e) {
            Log::error('Logo deletion failed: ' . $e->getMessage());
            Notification::make()
                ->title(__('Failed to delete logo'))
                ->body(__('Please try again.'))
                ->danger()
                ->send();
        }
    }

    public function deleteCover()
    {
        try {
            $this->user->clearMediaCollection('cover');
            $this->existingCoverUrl = null;
            $this->reset(['cover', 'coverUrl', 'tempCoverUrl']);

            Notification::make()
                ->title(__('Cover image deleted successfully'))
                ->success()
                ->send();
        } catch (\Exception $e) {
            Log::error('Cover deletion failed: ' . $e->getMessage());
            Notification::make()
                ->title(__('Failed to delete cover image'))
                ->body(__('Please try again.'))
                ->danger()
                ->send();
        }
    }

    public function clearTemporaryLogo()
    {
        $this->reset(['logo', 'tempLogoUrl']);
        
        Notification::make()
            ->title(__('Temporary logo removed'))
            ->success()
            ->send();
    }

    public function clearTemporaryCover()
    {
        $this->reset(['cover', 'tempCoverUrl']);
        
        Notification::make()
            ->title(__('Temporary cover removed'))
            ->success()
            ->send();
    }

    public function saveChanges()
    {
        try {
            $hasChanges = false;

            // Handle logo upload
            if ($this->logo instanceof TemporaryUploadedFile && $this->tempLogoUrl) {
                $this->validate([
                    'logo' => 'required|image|mimes:jpeg,png,jpg,webp|max:5120',
                ]);

                $this->user->clearMediaCollection('logo');
                $this->user->addMedia($this->logo->path())->toMediaCollection('logo');
                $this->existingLogoUrl = $this->user->getFirstMediaUrl('logo', 'medium');
                $this->reset(['logo', 'tempLogoUrl']);
                $hasChanges = true;
            }
            // Handle logo URL upload
            elseif ($this->logoUrl) {
                $this->validate([
                    'logoUrl' => 'required|url',
                ]);

                $this->user->clearMediaCollection('logo');
                $this->user->addMediaFromUrl($this->logoUrl)->toMediaCollection('logo');
                $this->existingLogoUrl = $this->user->getFirstMediaUrl('logo', 'medium');
                $this->reset('logoUrl');
                $hasChanges = true;
            }

            // Handle cover upload
            if ($this->cover instanceof TemporaryUploadedFile) {
                $this->validate([
                    'cover' => 'required|image|mimes:jpeg,png,jpg,webp|max:5120',
                ]);

                $this->user->clearMediaCollection('cover');
                $this->user->addMedia($this->cover->path())->toMediaCollection('cover');
                $this->existingCoverUrl = $this->user->getFirstMediaUrl('cover', 'medium');
                $this->reset(['cover', 'tempCoverUrl']);
                $hasChanges = true;
            }
            // Handle cover URL upload
            elseif ($this->coverUrl) {
                $this->validate([
                    'coverUrl' => 'required|url',
                ]);

                $this->user->clearMediaCollection('cover');
                $this->user->addMediaFromUrl($this->coverUrl)->toMediaCollection('cover');
                $this->existingCoverUrl = $this->user->getFirstMediaUrl('cover', 'medium');
                $this->reset('coverUrl');
                $hasChanges = true;
            }

            if ($hasChanges) {
                Notification::make()
                    ->title(__('Changes saved successfully'))
                    ->success()
                    ->send();
            } else {
                Notification::make()
                    ->title(__('No changes to save'))
                    ->info()
                    ->send();
            }
        } catch (\Exception $e) {
            Log::error('Profile photo update failed: ' . $e->getMessage());
            Notification::make()
                ->title(__('Failed to save changes'))
                ->body(__('Error: ' . $e->getMessage()))
                ->danger()
                ->send();
        }
    }

    public function render()
    {
        return view('livewire.dashboard.profile.profile-photo');
    }
}
