<?php

namespace App\Livewire\Dashboard\Profile;

use App\Services\User\UserServiceService;
use Illuminate\Support\Facades\Auth;
use Livewire\Component;
use Livewire\Attributes\Layout;
use Livewire\Attributes\Rule;
use Filament\Notifications\Notification;
use Filament\Actions\Action;
use Filament\Actions\Concerns\InteractsWithActions;
use Filament\Actions\Contracts\HasActions;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Form;
use Filament\Forms\Components\TagsInput;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Actions\Action as FormAction;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Gate;

#[Layout('layouts.dashboard.dashboard-app')]
class ProfileService extends Component implements HasForms, HasActions
{
    use InteractsWithForms;
    use InteractsWithActions;

    #[Rule('required|min:3|max:191')]
    public string $service_title = '';

    #[Rule('required|min:10')]
    public string $service_description = '';

    public $services = [];
    public $editingService = null;

    // New properties for profile meta form
    public $profileMetaForm = [];

    public function mount(UserServiceService $userServiceService)
    {
        $this->loadServices($userServiceService);

        // Load profile meta data
        $this->loadProfileMeta();
    }

    public function loadProfileMeta()
    {
        $user = Auth::user();
        $profile = $user->profile;

        if (!$profile) {
            Notification::make()
                ->title('Error')
                ->body('User profile not found.')
                ->danger()
                ->send();
            return;
        }

        // Get meta values
        $supportServicesValue = $profile->getMeta('support_services');
        $whyChooseUsValue = $profile->getMeta('why_choose_us');
        $emergencyServiceEnabled = $profile->getMeta('emergency_service_enabled');
        $emergencyServiceTitle = $profile->getMeta('emergency_service_title');
        $emergencyServiceDescription = $profile->getMeta('emergency_service_description');
        $emergencyServiceNumber = $profile->getMeta('emergency_service_number');

        // If why_choose_us doesn't exist or is empty, initialize with a default empty item
        if (empty($whyChooseUsValue)) {
            $whyChooseUsValue = [
                ['key' => '', 'value' => '']
            ];
        }

        // Fill the form with data
        $this->profileMetaForm = [
            'support_services' => $supportServicesValue ?: [],
            'why_choose_us' => $whyChooseUsValue ?: [],
            'emergency_service_enabled' => (bool)$emergencyServiceEnabled,
            'emergency_service_title' => $emergencyServiceTitle ?: '',
            'emergency_service_description' => $emergencyServiceDescription ?: '',
            'emergency_service_number' => $emergencyServiceNumber ?: '',
        ];

        // Fill the form with the data we've prepared
        $this->form->fill($this->profileMetaForm);
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                TagsInput::make('support_services')
                    ->rules(['max:5'])
                    ->label('Supported Services')
                    ->placeholder('Add a service and press Enter')
                    ->helperText('List the services you specialize in or support')
                    ->columnSpanFull()
                    ->visible(fn() => Gate::allows('userServiceSupportedServices')),

                Repeater::make('why_choose_us')
                    ->label('Why Choose Us Points')
                    ->schema([
                        TextInput::make('key')
                            ->label('Title')
                            ->placeholder('e.g., "Expert Technicians"')
                            ->maxLength(30)
                            ->required(),
                        TextInput::make('value')
                            ->label('Description')
                            ->placeholder('e.g., "Our team has over 15 years of industry experience"')
                            ->maxLength(200)
                            ->rules(['required', 'string', 'max:200'])
                            ->required(),
                    ])
                    ->columns(2)
                    ->addActionLabel('Add Selling Point')
                    ->deleteAction(
                        fn(FormAction $action) => $action->requiresConfirmation()
                    )
                    ->columnSpanFull()
                    ->helperText('Add unique selling points to highlight why clients should choose your services')
                    ->collapsible()
                    ->defaultItems(false)
                    ->visible(fn() => Gate::allows('userServiceWhyChooseUs')),

                // Emergency Service Section
                \Filament\Forms\Components\Section::make('Emergency Services')
                    ->schema([
                        \Filament\Forms\Components\Checkbox::make('emergency_service_enabled')
                            ->label('Offer Emergency Services')
                            ->helperText('Enable if you provide emergency services')
                            ->live(),

                        TextInput::make('emergency_service_title')
                            ->label('Emergency Service Title')
                            ->placeholder('e.g., "24/7 Emergency Support"')
                            ->maxLength(100)
                            ->hidden(fn(callable $get) => !$get('emergency_service_enabled'))
                            ->required(fn(callable $get) => $get('emergency_service_enabled')),

                        // \Filament\Forms\Components\Textarea::make('emergency_service_description')
                        //     ->label('Emergency Service Description')
                        //     ->placeholder('e.g., "We provide round-the-clock emergency support for critical issues"')
                        //     ->maxLength(200)
                        //     ->rows(3)
                        //     ->hidden(fn(callable $get) => !$get('emergency_service_enabled'))
                        //     ->required(fn(callable $get) => $get('emergency_service_enabled')),

                        TextInput::make('emergency_service_number')
                            ->label('Emergency Contact Number')
                            ->mask('+1 (999) 999-9999')
                            ->maxLength(20)
                            ->hidden(fn(callable $get) => !$get('emergency_service_enabled'))
                            ->required(fn(callable $get) => $get('emergency_service_enabled')),
                    ])
                    ->collapsible()
                    ->visible(fn() => Gate::allows('userServiceEmergencyServices')),
            ])
            ->statePath('profileMetaForm');
    }

    public function saveProfileMeta()
    {
        $data = $this->form->getState();
        $user = Auth::user();
        $profile = $user->profile;

        if (!$profile) {
            Notification::make()
                ->title('Error')
                ->body('User profile not found.')
                ->danger()
                ->send();
            return;
        }

        try {
            DB::beginTransaction();

            // Save support_services to user_profile_meta
            $profile->setMeta('support_services', $data['support_services']);

            // Save why_choose_us to user_profile_meta
            if (!empty($data['why_choose_us'])) {
                $profile->setMeta('why_choose_us', $data['why_choose_us']);
            } else {
                $profile->setMeta('why_choose_us', []);
            }

            // Save emergency service data
            $profile->setMeta('emergency_service_enabled', $data['emergency_service_enabled']);

            // Only save detail fields if emergency service is enabled
            if ($data['emergency_service_enabled']) {
                $profile->setMeta('emergency_service_title', $data['emergency_service_title'] ?? '');
                $profile->setMeta('emergency_service_description', $data['emergency_service_description'] ?? '');
                $profile->setMeta('emergency_service_number', $data['emergency_service_number'] ?? '');
            } else {
                // If emergency service is disabled, clear these fields
                // $profile->setMeta('emergency_service_title', '');
                // $profile->setMeta('emergency_service_description', '');
                // $profile->setMeta('emergency_service_number', '');
            }

            $profile->save();

            DB::commit();

            // After saving, reload the form data
            $this->loadProfileMeta();

            Notification::make()
                ->title('Profile meta saved successfully')
                ->success()
                ->send();
        } catch (\Exception $e) {
            DB::rollBack();

            Notification::make()
                ->title('Error saving profile meta')
                ->body('An error occurred while saving your profile meta data: ' . $e->getMessage())
                ->danger()
                ->send();
        }
    }

    public function loadServices(UserServiceService $userServiceService)
    {
        $this->services = $userServiceService->getUserServices(Auth::id())
            ->orderBy('service_order', 'asc')
            ->orderBy('created_at', 'desc')
            ->get();
    }

    public function deleteAction(): Action
    {
        return Action::make('delete')
            ->requiresConfirmation()
            ->color('danger')
            ->button()
            ->label('Remove')
            ->modalIcon('heroicon-o-trash')
            ->modalIconColor('danger')
            ->modalHeading('Delete Service')
            ->modalDescription('Are you sure you want to delete this service? This action cannot be undone.')
            ->modalSubmitActionLabel('Yes, delete it')
            ->modalCancelActionLabel('No, cancel')
            ->mountUsing(fn(Action $action, array $arguments) => $action->arguments(['uuid' => $arguments['uuid']]))
            ->action(function (Action $action) {
                $arguments = $action->getArguments();
                if (isset($arguments['uuid'])) {
                    $this->deleteService($arguments['uuid']);
                }
            });
    }

    public function save(UserServiceService $userServiceService)
    {
        $this->validate();

        $data = [
            'user_id' => Auth::id(),
            'service_title' => $this->service_title,
            'service_description' => $this->service_description,
            'created_by' => Auth::id(),
        ];

        try {
            if ($this->editingService) {
                $userServiceService->updateByUuid($this->editingService['uuid'], $data);

                Notification::make()
                    ->title('Service updated successfully')
                    ->success()
                    ->send();
            } else {
                $userServiceService->create($data);

                Notification::make()
                    ->title('Service created successfully')
                    ->success()
                    ->send();
            }

            $this->resetForm();
            $this->loadServices($userServiceService);
            $this->dispatch('close-modal');
        } catch (\Exception $e) {
            Notification::make()
                ->title('Error')
                ->body('There was an error processing your request.')
                ->danger()
                ->send();
        }
    }

    public function editService(string $uuid, UserServiceService $userServiceService)
    {
        try {
            $this->editingService = $userServiceService->findByUuid($uuid);
            $this->service_title = $this->editingService->service_title;
            $this->service_description = $this->editingService->service_description;

            // Dispatch event to open modal
            $this->dispatch('open-modal', 'add-service-modal');
        } catch (\Exception $e) {
            Notification::make()
                ->title('Error')
                ->body('Service not found.')
                ->danger()
                ->send();
        }
    }

    protected function deleteService(string $uuid)
    {
        try {
            $userServiceService = app(UserServiceService::class);
            $service = $userServiceService->findByUuid($uuid);

            if ($service) {
                $userServiceService->delete($service->id);

                Notification::make()
                    ->title('Service deleted successfully')
                    ->success()
                    ->send();

                $this->loadServices($userServiceService);
            }
        } catch (\Exception $e) {
            Notification::make()
                ->title('Error')
                ->body('There was an error deleting the service.')
                ->danger()
                ->send();
        }
    }

    public function resetForm(): void
    {
        $this->reset(['service_title', 'service_description', 'editingService']);
    }

    public function reorder($items): void
    {
        try {
            // Validate items array
            if (!is_array($items)) {
                throw new \Exception('Invalid data format');
            }

            // Begin transaction
            DB::beginTransaction();

            // Get all services first
            $userServiceService = app(UserServiceService::class);
            $services = collect($items)->map(function ($item) use ($userServiceService) {
                return [
                    'uuid' => $item['value'],
                    'service' => $userServiceService->findByUuid($item['value'])
                ];
            })->filter(function ($item) {
                return !is_null($item['service']);
            });

            // Update all services with new order
            foreach ($services as $index => $item) {
                $item['service']->update([
                    'service_order' => $index + 1
                ]);
            }

            // Commit transaction
            DB::commit();

            // Refresh the services list
            $this->loadServices($userServiceService);

            Notification::make()
                ->title('Order updated successfully')
                ->success()
                ->send();
        } catch (\Exception $e) {
            // Rollback transaction
            DB::rollBack();

            Notification::make()
                ->title('Error')
                ->body('There was an error updating the order.')
                ->danger()
                ->send();
        }
    }

    public function render()
    {
        return view('livewire.dashboard.profile.profile-service');
    }
}
