<?php

namespace App\Livewire\Dashboard\Project;

use App\Enums\ProjectStatus;
use App\Enums\ProjectCompletedStatus;
use App\Models\Project\Project;
use App\Models\Location\Country;
use App\Models\Location\State;
use App\Models\Location\City;
use App\Services\Project\ProjectService;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Form;
use Livewire\Component;
use Livewire\Attributes\Layout;
use App\Filament\Dashboard\Project\ProjectForm;
use Filament\Notifications\Notification;
use Illuminate\Support\Facades\Auth;

#[Layout('layouts.dashboard.dashboard-app')]
class CreateProject extends Component implements HasForms
{
    use InteractsWithForms;

    public ?array $data = [];
    public ?string $action = null;

    protected ProjectService $projectService;

    public function boot(ProjectService $projectService)
    {
        $this->projectService = $projectService;
    }

    public function mount(): void
    {
        $this->form->fill([
            'project_status' => ProjectStatus::Pending->value,
            'completed_status' => ProjectCompletedStatus::InProgress->value,
            'created_by' => auth()->guard('web')->user()?->id,
        ]);
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema(
                (new ProjectForm())->getForm('create')
            )
            ->model(Project::class)
            ->statePath('data');
    }


    public function saveAsDraft(): void
    {
        $this->action = 'draft';
        $this->create();
    }

    protected function rules()
    {
        return [
            'data.title' => 'required|alpha:ascii|min:3|max:100',
            'data.slug' => 'required|unique:projects,slug',
            'data.description' => 'required|min:10',
        ];
    }

    protected function messages()
    {
        return [
            'data.title.required' => 'The project title is required.',
            'data.title.regex' => 'The title must contain only letters (A-Z, a-z), periods (.), and spaces without numbers.',
            'data.title.min' => 'The project title must be at least :min characters.',
            'data.title.max' => 'The project title cannot exceed :max characters.',
            'data.slug.required' => 'The project slug is required.',
            'data.slug.unique' => 'This slug is already in use.',
            'data.description.required' => 'The project description is required.',
            'data.description.min' => 'The description must be at least :min characters.',
        ];
    }

    protected function saveLocationData(array &$data): void
    {
        // Create country record if needed
        if (!empty($data['country_name']) && empty($data['country_id'])) {
            $country = \App\Models\Location\Country::firstOrCreate(
                ['country_name' => $data['country_name']],
                [
                    'country_code' => $data['country_code'] ?? strtoupper(substr($data['country_name'], 0, 2)),
                    'created_by' => Auth::id(),
                ]
            );
            $data['country_id'] = $country->id;

            // Create state record if needed
            if (!empty($data['state_name']) && empty($data['state_id'])) {
                $state = \App\Models\Location\State::firstOrCreate(
                    [
                        'state_name' => $data['state_name'],
                        'country_id' => $country->id
                    ],
                    [
                        'created_by' => Auth::id(),
                    ]
                );
                $data['state_id'] = $state->id;

                // Create city record if needed
                if (!empty($data['city_name']) && empty($data['city_id'])) {
                    $city = \App\Models\Location\City::firstOrCreate(
                        [
                            'city_name' => $data['city_name'],
                            'state_id' => $state->id,
                            'country_id' => $country->id
                        ],
                        [
                            'created_by' => Auth::id(),
                        ]
                    );
                    $data['city_id'] = $city->id;
                }
            }
        }
    }

    public function create(): void
    {
        // Validate the form data using the defined rules
        $this->validate();

        $data = $this->form->getState();

        // Save location data before creating the project
        $this->saveLocationData($data);

        $data['project_status'] = match ($this->action) {
            'draft' => ProjectStatus::Draft->value,
            default => ProjectStatus::Pending->value
        };

        $data['user_id'] = auth()->guard('web')->user()?->id;
        $project = $this->projectService->create($data);
        $this->form->model($project)->saveRelationships();

        // Show appropriate notification
        Notification::make()
            ->success()
            ->title(match ($this->action) {
                'draft' => 'Saved as Draft',
                default => 'Submitted for Review'
            })
            ->body(match ($this->action) {
                'draft' => 'Project has been saved as draft successfully.',
                default => 'Project has been submitted for review.'
            })
            ->duration(5000)
            ->send();

        $this->redirect(route('dashboard.project.edit', ['uuid' => $project->uuid]), navigate: true);
    }

    public function cancel(): void
    {
        $this->redirect(route('dashboard.project'), navigate: true);
    }

    public function render()
    {
        return view('livewire.dashboard.project.create-project');
    }
}
