<?php

namespace App\Livewire\Dashboard\Project;

use App\Services\Project\ProjectService;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Form;
use Livewire\Component;
use Livewire\Attributes\Layout;
use App\Filament\Dashboard\Project\ProjectForm;
use App\Enums\ProjectStatus;
use App\Models\Project\Project;
use Filament\Notifications\Notification;

#[Layout('layouts.dashboard.dashboard-app')]

class EditProject extends Component implements HasForms
{
    use InteractsWithForms;

    public ?array $data = [];
    public Project $project;

    protected ProjectService $projectService;

    public function boot(ProjectService $projectService)
    {
        $this->projectService = $projectService;
    }

    public function mount(string $uuid): void
    {
        $this->project = $this->projectService->findByUuid($uuid);
        if (!$this->project) {
            abort(404);
        }

        $this->form->fill([
            'title' => $this->project->title,
            'user_id' => $this->project->user_id,
            'parent_id' => $this->project->parent_id,
            'slug' => $this->project->slug,
            'description' => $this->project->description,
            'techniques' => $this->project->techniques,
            'equipments' => $this->project->equipments,
            'introduction' => $this->project->introduction,
            'objective' => $this->project->objective,
            'challenges' => $this->project->challenges,
            'solutions' => $this->project->solutions,
            'outcomes' => $this->project->outcomes,
            'completed_status' => $this->project->completed_status,
            'completion_date' => $this->project->completion_date,
            'project_cost' => $this->project->project_cost,
            'tags' => $this->project->tags,
            'metrics' => $this->project->metrics,
            'project_status' => $this->project->project_status,
            'created_by' => $this->project->created_by,
            'published_at' => $this->project->published_at,
            'address' => $this->project->address,
            'location' => $this->project->location,
            'city_id' => $this->project->city_id,
            'state_id' => $this->project->state_id,
            'country_id' => $this->project->country_id,
            'country_name' => $this->project->country?->country_name,
            'state_name' => $this->project->state?->state_name,
            'city_name' => $this->project->city?->city_name,
            'postal_code' => $this->project->postal_code,
            'longitude' => $this->project->longitude,
            'latitude' => $this->project->latitude,
        ]);
        
        // The address field will be properly set by the map component's afterStateUpdated callback
        // when location is updated, so we don't need to set it here to avoid overwriting.
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema(
                (new ProjectForm())->getForm()
            )
            ->model($this->project ?? Project::class)
            ->statePath('data');
    }

    public function saveAsDraft(): void
    {
        $formData = $this->form->getState();
        $formData['project_status'] = ProjectStatus::Draft->value;

        // Update post using PostService with UUID
        $this->projectService->updateByUuid($this->project->uuid, $formData);

        $this->form->model($this->project)->saveRelationships();

        Notification::make()
            ->success()
            ->title('Project Updated')
            ->body('Your project has been saved as draft.')
            ->duration(5000)
            ->send();
    }

    public function save(): void
    {
        $formData = $this->form->getState();

        // Set status to pending for admin review
        $formData['project_status'] = ProjectStatus::Pending->value;

        // Update project using ProjectService with UUID
        $this->projectService->updateByUuid($this->project->uuid, $formData);

        $this->form->model($this->project)->saveRelationships();

        Notification::make()
            ->success()
            ->title('Project Updated')
            ->body('Your project has been updated and submitted for review.')
            ->duration(5000)
            ->send();
    }

    public function cancel(): void
    {
        $this->redirect(route('dashboard.project'), navigate: true);
    }
    public function render()
    {
        return view('livewire.dashboard.project.edit-project');
    }
}
