<?php

namespace App\Livewire\Dashboard\Team;

use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Form;
use Livewire\Component;
use Livewire\Attributes\Layout;
use App\Filament\Dashboard\Team\TeamForm;
use Filament\Notifications\Notification;
use App\Enums\TeamMemberStatus;
use App\Models\Team\TeamMember;
use App\Services\Team\TeamMemberService;

#[Layout('layouts.dashboard.dashboard-app')]

class CreateTeam extends Component implements HasForms
{

    use InteractsWithForms;

    public ?array $data = [];
    public ?string $action = null;

    protected TeamMemberService $teamMemberService;

    public function boot(TeamMemberService $teamMemberService)
    {
        $this->teamMemberService = $teamMemberService;
    }

    public function mount(): void
    {
        $this->form->fill([
            'project_status' => TeamMemberStatus::Pending->value,
            'created_by' => auth()->guard('web')->user()?->id,
        ]);
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema(
                (new TeamForm())->getForm('create')
            )
            ->model(TeamMember::class)
            ->statePath('data');
    }

    public function saveAsDraft(): void
    {
        $this->action = 'draft';
        $this->create();   
    }

    public function create(): void
    {
        $data = $this->form->getState();
        $data['team_member_status'] = match($this->action) {
            'draft' => TeamMemberStatus::Draft->value,
            default => TeamMemberStatus::Pending->value
        };

        $data['user_id'] = auth()->guard('web')->user()?->id;
        dump($data);
        $teamMember = $this->teamMemberService->create($data);
        $this->form->model($teamMember)->saveRelationships();

        // Show appropriate notification
        Notification::make()
            ->success()
            ->title(match($this->action) {
                'draft' => 'Saved as Draft',
                default => 'Submitted for Review'
            })
            ->body(match($this->action) {
                'draft' => 'Team has been saved as draft successfully.',
                default => 'Team has been submitted for review.'
            })
            ->duration(5000)
            ->send();

        $this->redirect(route('dashboard.team.edit', ['uuid' => $teamMember->uuid]), navigate: true);
    }

    public function cancel(): void
    {
        $this->redirect(route('dashboard.team'), navigate: true);
    }

    public function render()
    {
        return view('livewire.dashboard.team.create-team');
    }
}
