<?php

namespace App\Livewire\Dashboard\Team;

use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Form;
use Livewire\Component;
use Livewire\Attributes\Layout;
use App\Filament\Dashboard\Team\TeamForm;
use Filament\Notifications\Notification;
use App\Enums\TeamMemberStatus;
use App\Models\Team\TeamMember;
use App\Services\Team\TeamMemberService;

#[Layout('layouts.dashboard.dashboard-app')]

class EditTeam extends Component implements HasForms
{
    use InteractsWithForms;

    public ?array $data = [];
    public TeamMember $teamMember;

    protected TeamMemberService $teamMemberService;

    public function boot(TeamMemberService $teamMemberService)
    {
        $this->teamMemberService = $teamMemberService;
    }

    public function mount(string $uuid): void
    {
        $this->teamMember = $this->teamMemberService->findByUuid($uuid);
        
        if (!$this->teamMember) {
            abort(404);
        }

        $this->form->fill([
           
        'user_id'  => $this->teamMember->user_id,
        'first_name'=> $this->teamMember->first_name,
        'last_name' => $this->teamMember->last_name,
        'phone' => $this->teamMember->phone,
        'email' => $this->teamMember->email,
        'team_member_role' => $this->teamMember->team_member_role,
        'experience_years' => $this->teamMember->experience_years,
        'credentials' => $this->teamMember->credentials,
        'description' => $this->teamMember->description,
        'address' => $this->teamMember->address,
        'social_links' => $this->teamMember->social_links,
        'team_member_status' => $this->teamMember->team_member_status,
        'published_at' => $this->teamMember->published_at,
        'created_by' => $this->teamMember->created_by

        ]);
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema(
                (new TeamForm())->getForm()
            )
            ->model($this->teamMember ?? TeamMember::class)
            ->statePath('data');
    }

    public function saveAsDraft(): void
    {
        $formData = $this->form->getState();
        $formData['team_member_status'] = TeamMemberStatus::Draft->value;
            
            // Update post using PostService with UUID
            $this->teamMemberService->updateByUuid($this->teamMember->uuid, $formData);

            $this->form->model($this->teamMember)->saveRelationships();

            Notification::make()
                ->success()
                ->title('Team Updated')
            ->body('Your team has been saved as draft.')
                ->duration(5000)
                ->send();
    }

    public function save(): void
    {
        $formData = $this->form->getState();
        
        // Set status to pending for admin review
        $formData['team_member_status'] = TeamMemberStatus::Pending->value;
        
        // Update team using TeamMemberService with UUID
        $this->teamMemberService->updateByUuid($this->teamMember->uuid, $formData);

        $this->form->model($this->teamMember)->saveRelationships();

        Notification::make()
            ->success()
            ->title('Team Updated')
            ->body('Your team has been updated and submitted for review.')
            ->duration(5000)
            ->send();
    }

    public function cancel(): void
    {
        $this->redirect(route('dashboard.team'), navigate: true);
    }
    public function render()
    {
        return view('livewire.dashboard.team.edit-team');
    }
}
