<?php

namespace App\Livewire\Dashboard\Videos;

use App\Models\User\UserVideo;
use App\Services\User\UserVideoService;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Form;
use Livewire\Component;
use Livewire\Attributes\Layout;
use App\Filament\Dashboard\Video\VideoForm;
use Filament\Notifications\Notification;
use Illuminate\Support\Facades\DB;

#[Layout('layouts.dashboard.dashboard-app')]
class EditVideo extends Component implements HasForms
{
    use InteractsWithForms;

    public ?array $data = [];
    public UserVideo $video;

    protected UserVideoService $videoService;

    public function boot(UserVideoService $videoService)
    {
        $this->videoService = $videoService;
    }

    public function mount(string $uuid): void
    {
        $this->video = $this->videoService->findByUuid($uuid);
        if (!$this->video) {
            abort(404);
        }

        $this->form->fill([
            'video_title' => $this->video->video_title,
            'video_description' => $this->video->video_description,
            'video_link' => $this->video->video_link,
            'is_active' => $this->video->is_active,
            'is_featured' => $this->video->is_featured,
            'tags' => $this->video->tags,
        ]);
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema(
                (new VideoForm())->getForm('edit')
            )
            ->model($this->video ?? UserVideo::class)
            ->statePath('data');
    }

    public function update(): void
    {
        $formData = $this->form->getState();
        try {
            DB::beginTransaction();

            // Update video using VideoService with UUID directly
            $updated = $this->videoService->update($this->video->id, $formData);

            if ($updated) {
                $this->form->model($this->video)->saveRelationships();
            }

            DB::commit();

            Notification::make()
                ->success()
                ->title('Video Updated')
                ->body('Your video has been updated successfully.')
                ->duration(5000)
                ->send();

            $this->redirect(route('dashboard.video'), navigate: true);
        } catch (\Exception $e) {
            DB::rollBack();

            Notification::make()
                ->danger()
                ->title('Error')
                ->body('An error occurred while updating the video. Please try again.')
                ->duration(5000)
                ->send();

            report($e); // Log the error
        }
    }

    public function cancel(): void
    {
        $this->redirect(route('dashboard.video'), navigate: true);
    }

    public function render()
    {
        return view('livewire.dashboard.videos.edit-video');
    }
}
