<?php

declare(strict_types=1);

namespace App\Livewire\Frontend\Auth\Modal;

use Illuminate\Support\Facades\Auth;
use LivewireUI\Modal\ModalComponent;
use Livewire\Attributes\Validate;
use SnipifyDev\LaravelCaptcha\Rules\RecaptchaValidationRule;
use App\Enums\UserProfileStatus;

class LoginModal extends ModalComponent
{
    /** @var string */
    #[Validate('required|email')]
    public $email = '';

    /** @var string */
    #[Validate('required')]
    public $password = '';

    /** @var bool */
    #[Validate('boolean')]
    public $remember = false;

    /** @var bool */
    public $showPassword = false;

    /** @var string - reCAPTCHA token */
    public $captchaToken = '';


    public function authenticate()
    {

        // Validate all fields including captcha using Laravel validation
        $this->validate([
            'email' => 'required|email|max:255',
            'password' => 'required|string|min:6|max:255',
            'remember' => 'boolean',
            'captchaToken' => ['required', new RecaptchaValidationRule('login', null, 'v3')],
        ]);

        // Attempt to authenticate the user
        if (!Auth::attempt(['email' => $this->email, 'password' => $this->password], $this->remember)) {
            $this->addError('email', 'The provided credentials do not match our records.');
            return;
        }

        // Check if user profile is active
        $user = Auth::user();
        if ($user->profile && $user->profile->status !== UserProfileStatus::Active) {
            Auth::logout();
            
            $statusMessage = match($user->profile->status) {
                UserProfileStatus::NotActive => 'Your account is not verified. Please check your email for the verification link.',
                UserProfileStatus::PendingApproval => 'Your account is pending approval. Please wait for admin approval.',
                UserProfileStatus::OnHold => 'Your account has been temporarily suspended. Please contact support.',
                UserProfileStatus::Incomplete => 'Your profile is incomplete. Please complete your profile information.',
                UserProfileStatus::Cancelled => 'Your account has been cancelled. Please contact support.',
                UserProfileStatus::PastDue => 'Your account has overdue payments. Please update your payment information.',
                default => 'Your account is not active. Please contact support.'
            };
            
            $this->addError('email', $statusMessage);
            return;
        }

        $this->closeModal();

        return redirect()->intended(route('home'));
    }

    /**
     * Toggle password visibility
     */
    public function togglePasswordVisibility()
    {
        $this->showPassword = !$this->showPassword;
    }

    /**
     * Supported: 'sm', 'md', 'lg', 'xl', '2xl', '3xl', '4xl', '5xl', '6xl', '7xl'
     */
    public static function modalMaxWidth(): string
    {
        return 'lg';
    }

    public static function closeModalOnEscape(): bool
    {
        return true;
    }

    public static function closeModalOnClickAway(): bool
    {
        return true;
    }

    public static function closeModalOnEscapeIsForceful(): bool
    {
        return false;
    }

    public function render()
    {
        return view('livewire.frontend.auth.modal.login-modal');
    }
}
