<?php

namespace App\Livewire\Frontend\Auth\Modal;

use LivewireUI\Modal\ModalComponent;
use Illuminate\Auth\Events\Registered;
use Illuminate\Support\Facades\Auth;
use App\Services\User\UserService;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use SnipifyDev\LaravelCaptcha\Rules\RecaptchaValidationRule;
use App\Enums\UserProfileStatus;

class RegisterModal extends ModalComponent
{
    /** @var int */
    public $currentStep = 1;

    /** @var string */
    public $accountType = '';

    /** @var string */
    public $role = 'Subscriber';
    /** @var string */
    public $name = '';

    /** @var string */
    public $email = '';

    /** @var string */
    public $password = '';

    /** @var string */
    public $passwordConfirmation = '';

    /** @var string */
    public $phone = '';

    /** @var UserService */
    protected $userService;

    /** @var bool */
    public $showPassword = false;

    /** @var bool */
    public $showPasswordConfirmation = false;

    /** @var string - reCAPTCHA token */
    public $captchaToken = '';

    public function  boot(UserService $userService)
    {
        $this->userService = $userService;
    }

    /**
     * Reset modal to step 1 when opened
     */
    public function mount()
    {
        $this->currentStep = 1;
        $this->accountType = '';
        $this->role = 'Subscriber';
    }

    /**
     * Toggle password visibility
     */
    public function togglePasswordVisibility()
    {
        $this->showPassword = !$this->showPassword;
    }

    /**
     * Toggle password confirmation visibility
     */
    public function togglePasswordConfirmationVisibility()
    {
        $this->showPasswordConfirmation = !$this->showPasswordConfirmation;
    }

    /**
     * Proceed to step 2 if account type is selected
     */
    public function nextStep()
    {
        $this->validate([
            'accountType' => ['required', 'in:homeowner,service-provider']
        ], [
            'accountType.required' => 'Please select an account type to continue.',
            'accountType.in' => 'Please select a valid account type.'
        ]);

        // Set role based on account type
        $this->role = $this->accountType === 'homeowner' ? 'Subscriber' : 'Service';

        $this->currentStep = 2;
    }

    /**
     * Go back to step 1
     */
    public function previousStep()
    {
        $this->currentStep = 1;
        $this->resetErrorBag();
    }

    public function register()
    {
        $fields = $this->validate([
            'name' => ['required', 'string', 'max:100'],
            'email' => ['required', 'email'],
            'phone' => ['nullable'],
            'password' => ['required', 'min:8', 'same:passwordConfirmation',],
            'role' => ['required', 'string'],
            'captchaToken' => ['required', new RecaptchaValidationRule('register', null, 'v3')],
        ]);
        try {
            DB::beginTransaction();
            $user = $this->userService->create($fields);
            if ($user) {
                // Create user profile with phone number and not_active status
                $user->profile()->create([
                    'first_name' => str_contains($fields['name'], ' ') ? explode(' ', $fields['name'])[0] : $fields['name'],
                    'last_name' => str_contains($fields['name'], ' ') ? implode(' ', array_slice(explode(' ', $fields['name']), 1)) : '',
                    'phone' => $fields['phone'],
                    'status' => UserProfileStatus::NotActive
                ]);
                $user->assignRole($this->role);
            }

            event(new Registered($user));
            DB::commit();
            $this->closeModal();
            $this->addSuccess('success', 'Registration successful! Please check your email to verify your account.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('User registration failed: ' . $e->getMessage());
            $this->addError('error', 'Failed to register user. Please try again.');
        }

        return redirect()->intended(route('home'));
    }
    /**
     * Supported: 'sm', 'md', 'lg', 'xl', '2xl', '3xl', '4xl', '5xl', '6xl', '7xl'
     */
    public static function modalMaxWidth(): string
    {
        return 'lg';
    }

    public static function closeModalOnEscape(): bool
    {
        return true;
    }

    public static function closeModalOnClickAway(): bool
    {
        return true;
    }

    public static function closeModalOnEscapeIsForceful(): bool
    {
        return false;
    }

    public function render()
    {
        return view('livewire.frontend.auth.modal.register-modal');
    }
}
