<?php

namespace App\Livewire\Frontend\Auth;

use Illuminate\Auth\Events\PasswordReset as PasswordResetEvent;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Password;
use Illuminate\Support\Str;
use Livewire\Component;
use Livewire\Attributes\Layout;
use Illuminate\Support\Facades\Auth;

#[Layout('layouts.app')]
class PasswordReset extends Component
{
    /** @var string */
    public $token;

    /** @var string */
    public $email;

    /** @var string */
    public $password;

    /** @var string */
    public $password_confirmation;

    public function mount($token)
    {
        $this->token = $token;
        // Clean up the email parameter by removing any query string
        $email = request()->query('email', '');
        $this->email = strstr($email, '?', true) ?: $email;
    }

    public function resetPassword()
    {
        $this->validate([
            'token' => 'required',
            'email' => 'required|email',
            'password' => 'required|min:8|same:password_confirmation',
        ]);

        $response = $this->broker()->reset(
            [
                'token' => $this->token,
                'email' => $this->email,
                'password' => $this->password,
                'password_confirmation' => $this->password_confirmation
            ],
            function ($user, $password) {
                $user->password = Hash::make($password);
                $user->setRememberToken(Str::random(60));
                $user->save();

                event(new PasswordResetEvent($user));
                $this->guard()->login($user);
            }
        );

        if ($response === Password::PASSWORD_RESET) {
            session()->flash('status', 'Your password has been reset successfully!');
            $this->dispatch('notify', [
                'type' => 'success',
                'message' => 'Your password has been reset successfully!'
            ]);
            $this->dispatch('openModal', [
                'component' => 'frontend.auth.modal.login-modal'
            ]);
            return redirect()->route('home');
        }

        $this->addError('email', trans($response));
    }

    /**
     * Get the broker to be used during password reset.
     *
     * @return \Illuminate\Contracts\Auth\PasswordBroker
     */
    public function broker()
    {
        return Password::broker();
    }

    /**
     * Get the guard to be used during password reset.
     *
     * @return \Illuminate\Contracts\Auth\StatefulGuard
     */
    protected function guard()
    {
        return Auth::guard();
    }
       
    public function render()
    {
        return view('livewire.frontend.auth.password-reset');
    }
}
