<?php

namespace App\Livewire\Frontend\Blog;

use App\Services\Blog\PostCategoryService;
use App\Services\Blog\PostService;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Route;
use Livewire\Attributes\Url;
use Livewire\Component;
use Livewire\WithPagination;

class PostSearch extends Component
{
    use WithPagination;

    protected $postService;

    protected $listItemVersion;

    public $perPage = 10;

    public $listItemComponent;

    public $openTechAndInnovation   = false;
    public $openEquipmentAndRepair  = false;
    public $openWaterDrilling       = false;
    public $openWaterSystemRepair   = false;
    public $openDrillingTechniques  = false;
    public $openSafetyAndCompliance = false;

    #[Url(as: 'q')]
    public string $search = '';

    public $sort = 'newest';

    public $category = [];

    public $tag;

    public $author;

    public $pageTitle = 'Latest News';

    public $sorting = false;

    public $sortColumn = 'created_at';

    public $sortOrder;
    public $order;

    public $sortDirection = 'desc';

    public Collection $sortList;

    public $currentRoute = 'journal.index';

    public $categoryModel = null;

    protected $postCategoryService;

    public $categoriesList = [];
    public $page;
    public $seoData;
    public $breadcrumbs;
    public $searchForTxt;
    public $sortText;

    public function boot(PostService $postService, PostCategoryService $postCategoryService,)
    {
        $this->postService         = $postService;
        $this->postCategoryService = $postCategoryService;
    }

    public function mount($page, $breadcrumbs)
    {
        $this->currentRoute = Route::currentRouteName();
        $this->page         = $page;
        $this->breadcrumbs  = $breadcrumbs;
        // Initialize from current request to persist on refresh
        $this->search         = request()->query('search', request()->query('q', $this->search));
        $this->tag            = request()->query('tag', $this->tag);
        $this->author         = request()->query('author', $this->author);
        $this->sort           = request()->query('sort', $this->sort);
        $this->category       = request()->input('category', $this->category);

        $this->categoriesList = $this->postCategoryService->getAll()->get();
        $this->getSortList();
        $this->parseSortOrder();
        $this->generateSearchForText();
        $this->parseSortOrder();
    }

    public function updated($property, $value) {
        $this->applySearch();
    }

    public function applySearch()
    {
        // Build search parameters using standard Laravel query array format
        $searchParams = [
            'search'   => $this->search ? $this->search : null,
            'category' => array_values(array_filter($this->category)) ?: null,
            'tag'      => $this->tag ?: null,
            'author'   => $this->author ?: null,
            'sort'     => $this->sort ?: null,
        ];
        // dump($searchParams);
        // Redirect to search route so URL contains category[] indices and persists on refresh
        return redirect()->route('journal.search', array_filter($searchParams, fn($v) => ! is_null($v)));
    }

    public function generateSearchForText()
    {
        $searchText = [];

        if (! empty($this->search)) {
            $searchText[] = $this->search;
        }

        if (! empty($this->category)) {
            $categories = $this->postCategoryService->getBySlug($this->category);
            foreach ($categories as $category) {
                $searchText[] = $category->name;
            }
        }

        $this->searchForTxt = implode(', ', array_filter($searchText));
    }

    public function applySortOrder()
    {
        $this->parseSortOrder();
        $this->applySearch();
    }

    public function parseSortOrder()
    {
        // Set default sorting if not already set
        if ($this->sort == 'created-desc') {
            $this->sortOrder = 'created_at';
            $this->order     = 'desc';
            $this->sortText  = 'Most Recent';
        } elseif ($this->sort == 'created-asc') {
            $this->sortOrder = 'created_at';
            $this->order     = 'asc';
            $this->sortText  = 'Most Recent';
        } elseif ($this->sort == 'feautred-first') {
            $this->sortOrder = 'is_featured';
            $this->order     = 'desc';
            $this->sortText  = 'Featured First';
        } elseif ($this->sort == 'latest-first') {
            $this->sortOrder = 'published_at';
            $this->order     = 'desc';
            $this->sortText  = 'Latest First';
        } elseif ($this->sort == 'most-popular') {
            $this->sortOrder = 'updated_at';
            $this->order     = 'desc';
            $this->sortText  = 'Most Popular';
        } else {
            $this->sortOrder = 'created_at';
            $this->order     = 'desc';
            $this->sortText  = 'Best Match';
        }
    }

    /**
     * Prepare search arguments for querying members
     *
     * @param  bool  $includeListArgs  Whether to include list-specific arguments (limit, order_by, etc.)
     * @return array Formatted search arguments
     */
    protected function prepareSearchArgs(bool $includeListArgs = false): array
    {
        // Base arguments used by both views
        $args = [
            'search'   => $this->search,
            'category' => $this->category,
            'tag'      => $this->tag,
            'author'   => $this->author,
        ];

        // Add list-specific arguments if requested
        if ($includeListArgs) {
            $args['limit']    = $this->perPage;
            $args['order_by'] = $this->sortOrder;
            $args['order']    = $this->order;
        }

        // Merge searchParams if they exist
        if (! empty($this->searchParams)) {
            $args = array_merge($args, $this->searchParams);
        }

        return $args;
    }
    public function render()
    {
        $args  = $this->prepareSearchArgs(true);
        $posts = $this->postService->getAll($args)
            ->published();
        $posts = $posts->paginate($this->perPage);
        // Ensure pagination keeps current query parameters
        $posts->appends(request()->query());
        return view('livewire.frontend.blog.post-search', compact('posts'));
    }

    public function getSortList()
    {
        $this->sortList = collect([
            'newest' => __('Newest'),
        ]);
    }
}
