<?php

namespace App\Livewire\Frontend\Classified;

use Livewire\Component;
use App\Enums\ClassifiedCondition;
use App\Enums\ClassifiedLabel;
use App\Enums\UserProfileType;
use App\Models\Location\Country;
use App\Services\Classified\{ClassifiedService,ClassifiedTypeService, ClassifiedCategoryService, ClassifiedDeliveryService};
use App\Services\Location\{CountryService, StateService, CityService};
use Illuminate\Support\Collection;
use Illuminate\Support\Str;
use Livewire\Attributes\On;
use Livewire\Attributes\Url;
use Livewire\Attributes\Validate;
use Livewire\WithPagination;
use App\Enums\ClassifiedStatus;

class ClassifiedList extends Component
{
    use WithPagination;

    // Services
    protected $classifiedService;
    protected $classifiedCategoryService;
    protected $countryService;
    protected $stateService;
    protected $cityService;
    protected $classifiedTypeService;
    protected $classifiedDeliveryService;

    // Properties
    public $page;
    public $viewType;
    public $searchParams = [];
    public $searchForTxt;

    // Lists for dropdowns
    public $mainCategoriesList;
    public $countryList;
    public $stateList;
    public $cityList;
    public $subCategoriesList;
    public $conditionsList;
    public $labelsList;
    public $radiusList;
    public $countryCode;
    public $typesList;
    public $deliveryList;
    public $itemAgeList;
    public $datePostedList;
    public $sellerTypeList;
    public $flashFilterList;
    public $classifiedStatus = [];

    // Collapse states
    public bool $openCategory = true;
    // public bool $openSubcategory = false;
    public bool $openCondition = true;
    public bool $openLabel = true;
    public bool $openLocation = true;
    public bool $openPrice = true;
    public bool $openClassifiedAdType = true;
    public bool $openFlashSale = true;
    public bool $openItemAge = true;
    public bool $openPricing = true;
    public bool $openSellerType = true;
    public bool $openDatePosted = true;
    public bool $openTransactionAndDelivery = true;
    public bool $openAdvancedFilters = true;

    #[Validate('numeric|min:1|max:500')]
    public $customRadius;

    // URL Parameters
    #[Url(as: 'q')]
    public $search;

    #[Url]
    public $category = [];

    #[Url]
    public $subCategory = [];

    #[Url]
    public $country = '';

    #[Url]
    public $state = '';

    #[Url]
    public $city = '';

    #[Url]
    public $radius = 10;

    #[Url]
    public $condition = [];

    #[Url]
    public $label = [];

    #[Url]
    public $location = '';

    #[Url]
    public $minPrice;

    #[Url]
    public $maxPrice;

    #[Url]
    public $sort = 'created-desc';

    #[Url]
    public $itemAge = '';

    #[Url]
    public $datePosted = '';

    #[Url]
    public $type = [];

    #[Url]
    public $sellerType = [];

    #[Url]
    public $delivery = [];

    #[Url]
    public $flashFilter = [];

    #[Url]
    public bool $flashOnly = false;

    #[Url]
    public bool $newItems = false;

    #[Url]
    public bool $localPickup = false;

    // Location coordinates
    public $lat;
    public $lng;

    public $sortOrder;
    public $order;
    public $sortText;
    public $perPage = 15;
    public $totalRecords;
    public $totalPages;
    public $currentPage;
    protected $classifieds;
    public $breadcrumbs;

    public function boot(
        ClassifiedService $classifiedService,
        CountryService $countryService,
        StateService $stateService,
        CityService $cityService,
        ClassifiedTypeService $classifiedTypeService,
        ClassifiedCategoryService $classifiedCategoryService,
        ClassifiedDeliveryService $classifiedDeliveryService
    ) {
        // Initialize services
        $this->classifiedService = $classifiedService;
        $this->countryService = $countryService;
        $this->stateService = $stateService;
        $this->cityService = $cityService;
        $this->classifiedCategoryService = $classifiedCategoryService;
        $this->classifiedTypeService = $classifiedTypeService;
        $this->classifiedDeliveryService = $classifiedDeliveryService;
    }

    public function mount($page, $breadcrumbs)
    {
        $this->page = $page;
        $this->breadcrumbs = $breadcrumbs;

        $this->countryList = collect();
        $this->stateList = collect();
        $this->cityList = collect();
        $this->subCategoriesList = collect();
        $this->conditionsList = collect();
        $this->labelsList = collect();
        $this->radiusList = collect();
        $this->typesList = collect();
        $this->deliveryList = collect();
        $this->itemAgeList = collect();
        $this->datePostedList = collect();
        $this->sellerTypeList = collect();
        $this->flashFilterList = collect();
        $this->classifiedStatus = [ClassifiedStatus::Published->value];

        // Load countries 
        $this->getCountryList();

        // Load states if country is selected
        if (!empty($this->country)) {
            $this->getStatesList();
        }

        // Load cities if states are selected
        if (!empty($this->state)) {
            $this->getCitiesList();
        }

        // Load types
        $this->getTypesList();

        // Load conditions
        $this->getConditionsList();

        // Load item age list
        $this->getItemAgeList();

        // Load date posted list
        $this->getDatePostedList();

        // Load main category list
        $this->getParentCategories();

        // Load sub category list
        $this->getChildCategories();

        // Load seller type list
        $this->getSellerTypeList();

        // Load delivery list
        $this->getDeliveryList();

        // Load flash filter list
        $this->getFlashFilterList();

        // Load radius list
        $this->getRadiusList();

        // Load labels list
        $this->getLabelsList();

        // Parse location string into lat/lng if provided
        $this->parseLocationString();

        // Set initial sort order
        $this->parseSortOrder();

        // Load additional data
        // $this->loadAdditionalData();
    }

    public function render()
    {
        $args = $this->prepareSearchArgs(true);

        $classifieds = $this->classifiedService->getClassifiedList($args);
        $this->totalRecords = $classifieds->total();
        $this->totalPages = $classifieds->lastPage();
        $this->currentPage = $classifieds->currentPage();

        return view('livewire.frontend.classified.classified-list', [
            'classifieds' => $classifieds,
        ]);
    }

    public function updated($property, $value)
    {
        if ($property === 'search') {
            $this->applySearch();
        }
        if ($property === 'category') {
            $this->getSubCategoryList();
            if (empty($this->category)) {
                $this->subCategory = [];
                $this->subCategoriesList = collect();
            }
            // Ensure categories are sorted
            $this->mainCategoriesList = $this->mainCategoriesList->sortBy('name');
            $this->applySearch();
        }

        if (Str::startsWith($property, 'subCategory.')) {
            if (empty($this->subCategory)) {
                $this->subCategoriesList = collect();
            }
            // Ensure subcategories are sorted
            $this->subCategoriesList = $this->subCategoriesList->sortBy('name');
            $this->applySearch();
        }

        if ($property === 'country') {
            if ($value != 'all') {
                $this->getStatesList();
                $countryModel = $this->countryList->where('slug', $value)->first();
                if ($countryModel) {
                    $this->countryCode = $countryModel->country_code;
                }
            } else {
                $this->countryCode = '';
            }
            // When country changes, reset state and city
            $this->state = '';
            $this->city = '';
            $this->lat = null;
            $this->lng = null;
            $this->location = '';
            $this->radius = 10;
            $this->customRadius = null;
            $this->applySearch();
        }

        if ($property === 'state') {
            $this->getCitiesList();
            if (empty($this->state)) {
                $this->city = '';
            }
            $this->applySearch();
        }

        if ($property === 'city') {
            if (empty($value)) {
                $this->lat = null;
                $this->lng = null;
                $this->location = '';
            }
            $this->applySearch();
        }

        // Handle updates to lat or lng
        if ($property === 'lat' || $property === 'lng') {
            if (!empty($this->lat) && !empty($this->lng)) {
                $this->location = $this->lat . ',' . $this->lng;
            } elseif (empty($this->lat) || empty($this->lng)) {
                $this->location = '';
            }
        }

        // Handle direct updates to location string
        if ($property === 'location') {
            $this->parseLocationString();
            $this->applySearch();
        }

        if ($property === 'radius') {
            if ($this->radius != 'custom') {
                $this->applySearch();
            }
        }

        if ($property === 'customRadius') {
            $this->applySearch();
        }

        if ($property === 'minPrice' || $property === 'maxPrice') {
            $this->applySearch();
        }

        if (Str::startsWith($property, 'label.')) {
            $this->applySearch();
        }

        if (Str::startsWith($property, 'type.')) {
            $this->applySearch();
        }

        if (Str::startsWith($property, 'condition.')) {
            $this->applySearch();
        }

        if ($property === 'itemAge') {
            $this->applySearch();
        }

        if ($property === 'datePosted') {
            $this->applySearch();
        }

        if (Str::startsWith($property, 'sellerType.')) {
            $this->applySearch();
        }

        if (Str::startsWith($property, 'delivery.')) {
            $this->applySearch();
        }

        if (Str::startsWith($property, 'flashFilter.')) {
            $this->applySearch();
        }

        if ($property === 'flashOnly') {
            $this->applySearch();
        }
    }

    public function resetFilters()
    {
        $this->reset([
            'search',
            'category',
            'subCategory',
            'type',
            'condition',
            'itemAge',
            'datePosted',
            'sellerType',
            'country',
            'state',
            'city',
            'radius',
            'lat',
            'lng',
            'location',
            'minPrice',
            'maxPrice',
            'label',
            'delivery',
            'flashFilter',
            'flashOnly',
            'newItems',
            // 'localPickup'
        ]);

        // Reset dependent lists
        $this->stateList = collect();
        $this->cityList = collect();
        $this->mainCategoriesList = collect();
        $this->subCategoriesList = collect();

        // Apply the reset filters
        $this->applySearch();
    }

    public function applySortOrder()
    {
        $this->parseSortOrder();
        $this->applySearch();
    }

    public function parseSortOrder()
    {
        if ($this->sort == 'created-desc') {
            $this->sortOrder = 'created_at';
            $this->order = 'desc';
            $this->sortText = 'Most Recent';
        } elseif ($this->sort == 'created-asc') {
            $this->sortOrder = 'created_at';
            $this->order = 'asc';
            $this->sortText = 'Oldest First';
        } elseif ($this->sort == 'price-desc') {
            $this->sortOrder = 'price';
            $this->order = 'desc';
            $this->sortText = 'Price: High to Low';
        } elseif ($this->sort == 'price-asc') {
            $this->sortOrder = 'price';
            $this->order = 'asc';
            $this->sortText = 'Price: Low to High';
        } else {
            $this->sortOrder = 'created_at';
            $this->order = 'desc';
            $this->sortText = 'Best Match';
        }
    }

    private function parseLocationString()
    {
        if (!empty($this->location)) {
            $splitLocation = explode(',', $this->location);
            if (count($splitLocation) >= 2) {
                $this->lat = trim($splitLocation[0]);
                $this->lng = trim($splitLocation[1]);
            }
        }
    }

    public function applySearch()
    {
        // If we have lat/lng but no location string, set it
        if (!empty($this->lat) && !empty($this->lng) && empty($this->location)) {
            $this->location = $this->lat . ',' . $this->lng;
        }

        // Build search parameters
        $searchParams = [
            'search' => $this->search,
            'condition' => $this->condition,
            'country' => $this->country,
            'state' => $this->state,
            'city' => $this->city,
            'location' => $this->location,
            'lat' => $this->lat,
            'lng' => $this->lng,
            'category' => $this->category,
            'subCategory' => $this->subCategory,
            'type' => $this->type,
            'itemAge' => $this->itemAge,
            'datePosted' => $this->datePosted,
            'sellerType' => $this->sellerType,
            'delivery' => $this->delivery,
            'flashFilter' => $this->flashFilter,
            'flashOnly' => $this->flashOnly,
            'minPrice' => $this->minPrice,
            'maxPrice' => $this->maxPrice,
            'radius' => $this->radius === 'custom' ? $this->customRadius : $this->radius,
            'label' => $this->label,
            'newItems' => $this->newItems,
            'localPickup' => $this->localPickup,
            'sort' => $this->sort
        ];

        // Generate search text
        $this->generateSearchForText();

        // Dispatch event to results component
        $this->dispatch('search-classifieds', $searchParams);
    }

    #[On('search-classifieds')]
    public function updateSearchParams($params)
    {
        $this->resetPage();
        $this->searchParams = $params;
    }

    public function getParentCategories()
    {
        $this->mainCategoriesList  = $this->classifiedCategoryService->getAll()->get();
    }

    /**
     * Helper method to get child categories for a list of parent categories
     * 
     * @param array|Collection $parentSlugs Slugs of parent categories
     * @param string $parentType Type of parent ('category' or 'subcategory')
     * @return Collection Collection of child categories
     */
    private function fetchChildrenForCategories($parentSlugs, string $parentType): Collection
    {
        // Create a collection to store child categories
        $allChildren = collect();

        // Convert to collection if not already
        $parentSlugs = collect($parentSlugs);

        // If parent slugs are provided
        if ($parentSlugs->isNotEmpty()) {
            // Process each parent slug
            foreach ($parentSlugs as $slug) {
                $parent = $this->classifiedCategoryService->findBySlug($slug);

                if ($parent) {
                    // Get children for this parent
                    $children = $this->classifiedCategoryService->getChildCategories($parent->id);

                    // Ensure children is a collection
                    $children = $children instanceof Collection ? $children : collect($children);

                    // Merge with main collection
                    $allChildren = $allChildren->merge($children);
                }
            }
        }

        // Return unique children sorted by name
        return $allChildren->unique('id')->values()->sortBy('name');
    }

    private function filterSelectedChildren($selectedSlugs, Collection $availableChildren): array
    {
        $childrenIdsToKeep = $availableChildren->pluck('id')->toArray();

        return collect($selectedSlugs)
            ->filter(function ($slug) use ($childrenIdsToKeep) {
                $child = $this->classifiedCategoryService->findBySlug($slug);
                return $child && in_array($child->id, $childrenIdsToKeep);
            })
            ->toArray();
    }

    public function getSubCategoryList()
    {
        // Get all possible child categories for selected parent categories
        $allChildCategories = $this->fetchChildrenForCategories($this->category, 'category');

        // Filter selected subcategories if we have any previous subcategories
        if ($this->subCategoriesList instanceof Collection && $this->subCategoriesList->isNotEmpty()) {
            // Filter selected subcategories to only include those with parents still selected
            $this->subCategory = $this->filterSelectedChildren($this->subCategory, $allChildCategories);
        }

        // Update the subcategories list and sort by name
        $this->subCategoriesList = $allChildCategories->sortBy('name');
    }

    public function getChildCategories()
    {
        $this->getSubCategoryList();
    }

    public function getCountryList()
    {
        $this->countryList = $this->countryService->getAll()->withCount('classifieds')->get();
    }

    public function getRadiusList()
    {
        $defaultRadiusList = [
            ['id' => 10, 'name' => '10 miles'],
            ['id' => 25, 'name' => '25 miles'],
            ['id' => 50, 'name' => '50 miles'],
            ['id' => 100, 'name' => '100 miles'],
            ['id' => 250, 'name' => '250 miles'],
            ['id' => 'custom', 'name' => 'Custom Radius'],
        ];
        $settingsRadiusList = get_general_setting('classified_search_radius_list', $defaultRadiusList);
        $this->radiusList = collect($settingsRadiusList);
    }

    public function getConditionsList()
    {
        $this->conditionsList = collect(ClassifiedCondition::cases());
    }

    public function getLabelsList()
    {
        $this->labelsList = collect(ClassifiedLabel::cases());
    }

    private function fetchStatesForCountry($countrySlug): Collection
    {
        $allStates = collect();

        if (is_array($countrySlug)) {
            $countrySlug = !empty($countrySlug) ? (string)reset($countrySlug) : '';
        }

        if (!empty($countrySlug) && is_string($countrySlug)) {
            $country = $this->countryService->getCountryBySlug($countrySlug);

            if ($country) {
                $states = $this->stateService->getAll(['country_id' => $country->id])->get();
                $allStates = $states instanceof Collection ? $states : collect($states);
            }
        }

        return $allStates->unique('id')->values();
    }

    private function fetchCitiesForStates($stateSlugs): Collection
    {
        $allCities = collect();
        $stateSlugs = collect($stateSlugs);

        if ($stateSlugs->isNotEmpty()) {
            foreach ($stateSlugs as $slug) {
                $state = $this->stateService->getStateBySlug($slug);

                if ($state) {
                    $cities = $this->cityService->getAll(['state_id' => $state->id])->get();
                    $cities = $cities instanceof Collection ? $cities : collect($cities);
                    $allCities = $allCities->merge($cities);
                }
            }
        }

        return $allCities->unique('id')->values();
    }

    private function filterSelectedLocations($selectedSlugs, Collection $availableLocations, string $type): array
    {
        $locationIdsToKeep = $availableLocations->pluck('id')->toArray();

        return collect($selectedSlugs)
            ->filter(function ($slug) use ($locationIdsToKeep, $type) {
                $location = null;
                if ($type === 'state') {
                    $location = $this->stateService->getStateBySlug($slug);
                } elseif ($type === 'city') {
                    $location = $this->cityService->getCityBySlug($slug);
                }

                return $location && in_array($location->id, $locationIdsToKeep);
            })
            ->toArray();
    }

    public function getStatesList()
    {
        $allStates = $this->fetchStatesForCountry($this->country);

        if ($this->stateList instanceof Collection && $this->stateList->isNotEmpty()) {
            if (empty($this->country) || $allStates->isEmpty()) {
                $this->state = '';
                $this->city = '';
                $this->stateList = collect();
                $this->cityList = collect();
            } else {
                $stateIdsToKeep = $allStates->pluck('id')->toArray();

                $currentState = is_array($this->state) ? $this->state : [$this->state];
                $filteredStates = collect($currentState)
                    ->filter(function ($stateSlug) use ($stateIdsToKeep) {
                        if (empty($stateSlug)) {
                            return false;
                        }
                        $state = $this->stateService->getStateBySlug($stateSlug);
                        return $state && in_array($state->id, $stateIdsToKeep);
                    })
                    ->toArray();

                $this->state = !is_array($this->state) && count($filteredStates) > 0 ? $filteredStates[0] : '';

                if (empty($this->state)) {
                    $this->city = '';
                    $this->cityList = collect();
                } else {
                    $this->getCitiesList();
                }
            }
        }

        $this->stateList = $allStates;
    }

    public function getCitiesList()
    {
        $stateArray = is_string($this->state) && !empty($this->state) ? [$this->state] : [];
        $allCities = $this->fetchCitiesForStates($stateArray);

        if (!empty($this->city) && $this->cityList instanceof Collection && $this->cityList->isNotEmpty()) {
            $currentCity = is_array($this->city) ? $this->city : [$this->city];
            $filteredCities = $this->filterSelectedLocations($currentCity, $allCities, 'city');
            $this->city = !is_array($this->city) && count($filteredCities) > 0 ? $filteredCities[0] : '';
        }

        $this->cityList = $allCities;
    }

    public function hydrate()
    {
        if (is_array($this->country)) {
            $this->country = !empty($this->country) ? (string)reset($this->country) : '';
        }

        if (is_array($this->state)) {
            $this->state = !empty($this->state) ? (string)reset($this->state) : '';
        }

        if (is_array($this->city)) {
            $this->city = !empty($this->city) ? (string)reset($this->city) : '';
        }

        $this->parseLocationString();
    }

    public function updatedCategory($value): void
    {
        if (!empty($value)) {
            $this->openSubcategory = true;
        }
    }

    public function toggle(string $section): void
    {
        $prop = 'open' . Str::studly($section);
        if (property_exists($this, $prop)) {
            $this->$prop = !$this->$prop;
        }
    }

    protected function prepareSearchArgs(bool $includeListArgs = true): array
    {
        $baseArgs = [
            'classified_status' => $this->classifiedStatus,
            'search' => $this->search,
            'condition' => $this->condition,
            'country' => $this->country,
            'state' => $this->state,
            'city' => $this->city,
            'location' => $this->location,
            'lat' => $this->lat,
            'lng' => $this->lng,
            'category' => $this->category,
            'subCategory' => $this->subCategory,
            'type' => $this->type,
            'itemAge' => $this->itemAge,
            'datePosted' => $this->datePosted,
            'sellerType' => $this->sellerType,
            'delivery' => $this->delivery,
            'flashFilter' => $this->flashFilter,
            'flashOnly' => $this->flashOnly,
        ];

        if ($includeListArgs) {
            $baseArgs['limit'] = $this->perPage;
            $baseArgs['order_by'] = $this->sortOrder;
            $baseArgs['order'] = $this->order;
        }

        // Add classified-specific arguments
        $classifiedArgs = [
            'type' => $this->type,
            'seller_type' => $this->sellerType,
            'delivery' => $this->delivery,
            'flash_filter' => $this->flashFilter,
            'flash_only' => $this->flashOnly,
            'new_items' => $this->newItems,
            'local_pickup' => $this->localPickup,
        ];

        $args = array_merge($baseArgs, $classifiedArgs);

        if (!empty($this->searchParams)) {
            $args = array_merge($args, $this->searchParams);
        }

        return $args;
    }

    public function generateSearchForText()
    {
        $searchText = '';
        $location = '';

        if (!empty($this->search)) {
            $searchText .= $this->search;
        }


        if (!empty($this->type)) {
            $typeList = [];
            foreach ($this->type as $key => $type) {
                $typeData = collect($this->typesList)->where('slug', $type)->first();
                if ($typeData && $key < 2) {
                    $typeList[] = $typeData->classified_type_name;
                }
                if (count($this->type) > 2) {
                    $searchText .= ' and ' . (count($this->type) - 2) . ' more';
                }
            }
        }

        if (!empty($this->city)) {
            $location .= ' ' . $this->city;
        } else {
            if (!empty($this->state)) {
                $stateData = $this->stateList->where('slug', $this->state)->first();
                if ($stateData) {
                    $location .= ' ' . $stateData->state_name;
                }
            }
            if (!empty($this->country)) {
                $countryCollection = collect($this->countryList);
                $countryData = $countryCollection->where('slug', $this->country)->first();
                if ($countryData) {
                    $location .= ' ' . $countryData->country_name;
                }
            }
        }

        if (!empty($location)) {
            $searchText .= ' near' . $location;
        }

        if (!empty($this->minPrice) || !empty($this->maxPrice)) {
            $searchText .= ' price: ';
            if (!empty($this->minPrice)) {
                $searchText .= '$' . $this->minPrice;
            }
            $searchText .= ' - ';
            if (!empty($this->maxPrice)) {
                $searchText .= '$' . $this->maxPrice;
            }
        }

        $this->searchForTxt = $searchText;
    }

    public function getTypesList()
    {
        $args = [
            'limit' => 10
        ];
        $this->typesList = $this->classifiedTypeService->getAll($args)->get();
    }

    public function getSellerTypeList()
    {
        $this->sellerTypeList = UserProfileType::cases();
    }

    public function getDeliveryList()
    {
        $this->deliveryList = $this->classifiedDeliveryService->getAll()->get();
    }

    public function getFlashFilterList()
    {
        $defaultFlashFilterList = [
            [
                'id' => 1,
                'name' => 'Ends in < 24 hours',
            ],
            [
                'id' => 2,
                'name' => '1–3 Days Left',
            ],
            [
                'id' => 3,
                'name' => '3–7 Days Left',
            ]
        ];

        $settingflashFilterList = get_general_setting('classified_flash_filter_list', $defaultFlashFilterList);
        $this->flashFilterList = collect($settingflashFilterList);
    }

    public function getItemAgeList()
    {
        $defaultItemAgeList = [
            [
                'id'  => 1,
                'value' => 'new',
                'name' => 'New (0 years)',
            ],
            [
                'id'  => 2,
                'value' => '0-1',
                'name' => 'Nearly New (< 1 year)',
            ],
            [
                'id'  => 3,
                'value' => '1-3',
                'name' => '1–3 Years',
            ],
            [
                'id'  => 4,
                'value' => '3-5',
                'name' => '3–5 Years',
            ],
            [
                'id'  => 5,
                'value' => '5-10',
                'name' => '5–10 Years',
            ],

            [
                'id'  => 6,
                'value' => '10-20',
                'name' => '10-20 Years',
            ],
            [
                'id'  => 7,
                'value' => '20-50',
                'name' => '20+ Years',
            ],
            [
                'id'  => 8,
                'value' => '50-100',
                'name' => 'Vintage',
            ],
            [
                'id'  => 9,
                'value' => '100+',
                'name' => 'Antique',
            ],
        ];

        $settingItemAgeList = get_general_setting('classified_item_age_list', $defaultItemAgeList);
        $this->itemAgeList = collect($settingItemAgeList);
    }

    public function getDatePostedList()
    {
        $defaultDatePostedList = [
            [
                'id'  => 1,
                'name' => 'Today',
            ],
            [
                'id'  => 3,
                'name' => 'Last 3 Days',
            ],
            [
                'id'  => 7,
                'name' => 'Last 7 Days',
            ],
            [
                'id'  => 30,
                'name' => 'Last 30 Days',
            ],

        ];

        $settingDatePostedList = get_general_setting('classified_date_posted_list', $defaultDatePostedList);
        $this->datePostedList = collect($settingDatePostedList);
    }
}
