<?php

namespace App\Livewire\Frontend\Classified\Partials;

use Livewire\Component;
use Livewire\Attributes\Url;
use App\Services\Location\CountryService;
use App\Services\Location\StateService;
use App\Services\Location\CityService;
use App\Services\Classified\ClassifiedTypeService;
use App\Services\Classified\ClassifiedCategoryService;
use App\Services\Classified\ClassifiedDeliveryService;
use Illuminate\Support\Collection;
use Illuminate\Support\Str;
use App\Models\Location\Country;
use App\Enums\UserProfileType;

class SearchClassifiedSidebarV1 extends Component
{
    protected $countryService;
    protected $stateService;
    protected $cityService;
    protected $classifiedTypeService;
    protected $classifiedCategoryService;
    protected $classifiedDeliveryService;

    public $openLocation = false;
    public $openClassifiedAdType = false;
    public $openFlashSale = false;
    public $openCategory = false;
    public $openCondition = false;
    public $openItemAge = false;
    public $openPricing = false;
    public $openSellerTypeAndDatePosted = false;
    public $openTransactionAndDelivery = false;
    public $openAdvancedFilters = false;
    public $countryCode;


    public $countryList = [];
    public $stateList = [];
    public $cityList = [];
    public $typeList = [];
    public $conditionList = [];
    public $itemAgeList = [];
    public $datePostedList = [];
    public $mainCategoryList = [];
    public $subCategoryList = [];
    public $sellerTypeList = [];
    public $deliveryList = [];
    public $flashFilterList = [];

    #[Url(as: 'q')]
    public $search;
    #[Url]
    public $country = '';
    #[Url]
    public $state = '';
    #[Url]
    public $city = '';
    #[Url]
    public $location = '';
    #[Url]
    public $type = [];
    #[Url]
    public $condition = '';
    #[Url]
    public $itemAge = '';
    #[Url]
    public $datePosted = '';
    #[Url]
    public $mainCategory = '';
    #[Url]
    public $subCategory = '';
    #[Url]
    public $sellerType = [];
    #[Url]
    public $delivery = [];
    #[Url]
    public $flashFilter = [];
    #[Url]
    public $flashOnly;

    // Latitude and longitude properties for location search
    public $lat;
    public $lng;

    /**
     * Boot the component
     *
     * @param CountryService $countryService
     * @param StateService $stateService
     * @param CityService $cityService
     * @param ClassifiedTypeService $classifiedTypeService
     */
    public function boot(CountryService $countryService, StateService $stateService, CityService $cityService, ClassifiedTypeService $classifiedTypeService, ClassifiedCategoryService $classifiedCategoryService, ClassifiedDeliveryService $classifiedDeliveryService)
    {
        $this->countryService = $countryService;
        $this->stateService = $stateService;
        $this->cityService = $cityService;
        $this->classifiedTypeService = $classifiedTypeService;
        $this->classifiedCategoryService = $classifiedCategoryService;
        $this->classifiedDeliveryService = $classifiedDeliveryService;
    }

    public function mount()
    {
        // Load countries 
        $this->getCountryList();


        // Load states if country is selected
        if (!empty($this->country)) {
            $this->getStatesList();

            // Load cities if states are selected
        }
        if (!empty($this->state)) {
            $this->getCitiesList();
        }
        // Load types
        $this->getTypesList();
        // Load conditions
        $this->getConditionList();

        // Load item age list
        $this->getItemAgeList();
        // Load date posted list
        $this->getDatePostedList();

        // Load main category list
        $this->getMainCategoryList();
        // Load sub category list
        $this->getSubCategoryList();

        // Load seller type list
        $this->getSellerTypeList();

        // Load delivery list
        $this->getDeliveryList();

        // Load flash filter list
        $this->getFlashFilterList();

        // Parse location string into lat/lng if provided
        $this->parseLocationString();
    }

    /**
     * Reset all filters to their default values
     */
    public function resetFilters()
    {
        $this->reset([
            'search',
            'mainCategory',
            'subCategory',
            'type',
            'condition',
            'itemAge',
            'datePosted',
            'sellerType',
            'country',
            'state',
            'city',
            'lat',
            'lng',
            'location',
            'flashOnly'
        ]);


        // Reset dependent lists
        $this->stateList = collect();
        $this->cityList = collect();
        $this->subCategoryList = collect();

        // Apply the reset filters
        $this->applySearch();
    }

    /**
     * Apply search filters and dispatch event to results component
     */
    public function applySearch()
    {
        // If we have lat/lng but no location string, set it
        if (!empty($this->lat) && !empty($this->lng) && empty($this->location)) {
            $this->location = $this->lat . ',' . $this->lng;
        }

        // Build search parameters
        $searchParams = [
            'search' => $this->search,
            'condition' => $this->condition,
            'country' => $this->country,
            'state' => $this->state,
            'city' => $this->city,
            'location' => $this->location,
            'lat' => $this->lat,
            'lng' => $this->lng,
            'mainCategory' => $this->mainCategory,
            'subCategory' => $this->subCategory,
            'type' => $this->type,
            'condition' => $this->condition,
            'itemAge' => $this->itemAge,
            'datePosted' => $this->datePosted,
            'sellerType' => $this->sellerType,
            'delivery' => $this->delivery,
            'flashFilter' => $this->flashFilter,
            'flashOnly' => $this->flashOnly,
        ];
        // Dispatch event to results component
        $this->dispatch('search-classifieds', $searchParams);
    }

    public function updated($property, $value)
    {
        if ($property === 'search') {
            $this->applySearch();
        }
        if (Str::startsWith($property, 'mainCategory')) {
            $this->getSubCategoryList();
            if (empty($this->mainCategory)) {
                $this->subCategory = [];
            }
            $this->applySearch();
        }

        if (Str::startsWith($property, 'subCategory')) {
            $this->applySearch();
        }

        if ($property === 'country') {

            if ($value != 'all') {
                $this->getStatesList();
                $countryModel = Country::where('slug', $value)->first();
                if ($countryModel) {
                    $this->countryCode = $countryModel->country_code;
                }
            } else {
                $this->countryCode = '';
            }
            // When country changes, reset state and city
            $this->state = [];
            $this->city = [];
            $this->lat = null;
            $this->lng = null;
            $this->location = '';
            $this->applySearch();
        }

        if (Str::startsWith($property, 'state')) {
            $this->getCitiesList();
            if (empty($this->state)) {
                $this->city = [];
            }
            $this->applySearch();
        }

        if ($property === 'city') {
            if (empty($value)) {
                $this->lat = null;
                $this->lng = null;
                $this->location = '';
            }
            $this->applySearch();
        }

        // Handle updates to lat or lng
        if ($property === 'lat' || $property === 'lng') {
            // Update location string when coordinates change
            if (!empty($this->lat) && !empty($this->lng)) {
                $this->location = $this->lat . ',' . $this->lng;
            } else if (empty($this->lat) || empty($this->lng)) {
                $this->location = '';
            }
        }

        // Handle direct updates to location string
        if ($property === 'location') {
            $this->parseLocationString();
            $this->applySearch();
        }

        // Apply search on other filter changes
        if (Str::startsWith($property, 'type.')) {
            $this->applySearch();
        }
        if ($property === 'condition') {
            $this->applySearch();
        }
        if ($property === 'itemAge') {
            $this->applySearch();
        }
        if ($property === 'datePosted') {
            $this->applySearch();
        }
        if ($property === 'flashFilter') {
            $this->applySearch();
        }
        if ($property === 'sellerType') {
            $this->applySearch();
        }
        if (Str::startsWith($property, 'delivery.')) {
            $this->applySearch();
        }
        if ($property === 'flashFilter') {
            $this->applySearch();
        }
        if ($property === 'flashOnly') {
            $this->applySearch();
        }
    }

    /**
     * Parse location string into lat/lng coordinates
     */
    private function parseLocationString()
    {
        if (!empty($this->location)) {
            $splitLocation = explode(',', $this->location);
            if (count($splitLocation) >= 2) {
                $this->lat = trim($splitLocation[0]);
                $this->lng = trim($splitLocation[1]);
            }
        }
    }
    /**
     * Get all states for selected country
     *
     * @param string|null $countrySlug Country slug to get states for
     * @return Collection Collection of states
     */
    private function fetchStatesForCountry($countrySlug): Collection
    {
        $allStates = collect();

        // Make sure countrySlug is a string
        if (is_array($countrySlug)) {
            // If somehow we got an array, take the first value
            $countrySlug = !empty($countrySlug) ? (string)reset($countrySlug) : '';
        }

        // If country is provided
        if (!empty($countrySlug) && is_string($countrySlug)) {
            $country = $this->countryService->getCountryBySlug($countrySlug);

            if ($country) {
                // Get states for this country
                $states = $this->stateService->getAll(['country_id' => $country->id])->get();

                // Ensure states is a collection
                $allStates = $states instanceof Collection ? $states : collect($states);
            }
        }

        // Return states
        return $allStates->unique('id')->values();
    }

    /**
     * Get all cities for selected states
     *
     * @param array|Collection $stateSlugs State slugs to get cities for
     * @return Collection Collection of cities
     */
    private function fetchCitiesForStates($stateSlugs): Collection
    {
        $allCities = collect();

        // Convert to collection if not already
        $stateSlugs = collect($stateSlugs);

        // If states are provided
        if ($stateSlugs->isNotEmpty()) {
            // Process each state slug
            foreach ($stateSlugs as $slug) {
                $state = $this->stateService->getStateBySlug($slug);

                if ($state) {
                    // Get cities for this state
                    $cities = $this->cityService->getAll(['state_id' => $state->id])->get();

                    // Ensure cities is a collection
                    $cities = $cities instanceof Collection ? $cities : collect($cities);

                    // Merge with main collection
                    $allCities = $allCities->merge($cities);
                }
            }
        }

        // Return unique cities
        return $allCities->unique('id')->values();
    }

    /**
     * Filter selected locations based on what's available in the full location list
     * 
     * @param array|Collection $selectedSlugs rrently selected location slugs
     * @param Collection $availableLocations Available locations to filter against
     * @param string $type Type of location ('state' or 'city')
     * @return array Filtered selection array
     */
    private function filterSelectedLocations($selectedSlugs, Collection $availableLocations, string $type): array
    {
        $locationIdsToKeep = $availableLocations->pluck('id')->toArray();

        return collect($selectedSlugs)
            ->filter(function ($slug) use ($locationIdsToKeep, $type) {
                $location = null;
                if ($type === 'state') {
                    $location = $this->stateService->getStateBySlug($slug);
                } elseif ($type === 'city') {
                    $location = $this->cityService->getCityBySlug($slug);
                }
                return $location && in_array($location->id, $locationIdsToKeep);
            })
            ->toArray();
    }

    /**
     * Helper method to get child categories for a list of parent categories
     * 
     * @param array|Collection $parentSlugs Slugs of parent categories
     * @param string $parentType Type of parent ('category' or 'subcategory')
     * @return Collection Collection of child categories
     */
    private function fetchChildrenForCategories($parentSlugs, string $parentType): Collection
    {
        // Create a collection to store child categories
        $allChildren = collect();

        // Convert to collection if not already
        $parentSlugs = collect($parentSlugs);

        // If parent slugs are provided
        if ($parentSlugs->isNotEmpty()) {
            // Process each parent slug
            foreach ($parentSlugs as $slug) {
                $parent = $this->classifiedCategoryService->findBySlug($slug);

                if ($parent) {
                    // Get children for this parent
                    $children = $this->classifiedCategoryService->getChildCategories($parent->id);

                    // Ensure children is a collection
                    $children = $children instanceof Collection ? $children : collect($children);

                    // Merge with main collection
                    $allChildren = $allChildren->merge($children);
                }
            }
        }

        // Return unique children sorted by name
        return $allChildren->unique('id')->values()->sortBy('name');
    }

    /**
     * Filter selected children based on what's available in the full children list
     * 
     * @param array|Collection $selectedSlugs Currently selected child slugs
     * @param Collection $availableChildren Available child categories to filter against
     * @return array Filtered selection array
     */
    private function filterSelectedChildren($selectedSlugs, Collection $availableChildren): array
    {
        $childrenIdsToKeep = $availableChildren->pluck('id')->toArray();

        return collect($selectedSlugs)
            ->filter(function ($slug) use ($childrenIdsToKeep) {
                $child = $this->classifiedCategoryService->findBySlug($slug);
                return $child && in_array($child->id, $childrenIdsToKeep);
            })
            ->toArray();
    }
    public function getCountryList()
    {
        $this->countryList = $this->countryService->getAll()->get();
    }

    public function getStatesList()
    {
        // Get all possible states for selected country
        $allStates = $this->fetchStatesForCountry($this->country);

        // Filter selected states if we have any previous states
        if ($this->stateList instanceof Collection && $this->stateList->isNotEmpty()) {
            // If country changed, check if states need to be filtered
            if (empty($this->country) || $allStates->isEmpty()) {
                // Country was removed or has no states, clear states and cities
                $this->state = [];
                $this->city = [];
                $this->stateList = collect();
                $this->cityList = collect();
            } else {
                // Filter selected states to only include those with parent country still selected
                $stateIdsToKeep = $allStates->pluck('id')->toArray();

                $this->state = collect((array)$this->state)
                    ->filter(function ($stateSlug) use ($stateIdsToKeep) {
                        $state = $this->stateService->getStateBySlug($stateSlug);
                        return $state && in_array($state->id, $stateIdsToKeep);
                    })
                    ->toArray();

                // If all states were removed, clear cities as well
                if (empty($this->state)) {
                    $this->city = [];
                    $this->cityList = collect();
                } else {
                    // Update cities based on remaining states
                    $this->getCitiesList();
                }
            }
        }

        // Update the state list
        $this->stateList = $allStates;
    }

    public function getCitiesList()
    {
        // Get all possible cities for selected states
        $allCities = $this->fetchCitiesForStates($this->state);

        // Filter selected cities if needed
        if (!empty($this->city) && $this->cityList instanceof Collection && $this->cityList->isNotEmpty()) {
            // Filter selected cities to only include those with parent states still selected
            $this->city = $this->filterSelectedLocations($this->city, $allCities, 'city');
        }

        // Update the city list
        $this->cityList = $allCities;
    }

    /**
     * Ensure properties are properly typed after Livewire hydration
     */
    public function hydrate()
    {
        // Ensure country is always a string, not an array
        if (is_array($this->country)) {
            $this->country = !empty($this->country) ? (string)reset($this->country) : '';
        }

        // Re-parse location after hydration
        $this->parseLocationString();
    }

    public function getTypesList()
    {
        $args = [
            'limit' => 10
        ];
        $this->typeList = $this->classifiedTypeService->getAll($args)->get();
    }

    public function getConditionList()
    {
        $this->conditionList = esp_get_classified_condition_list();
    }

    public function getItemAgeList()
    {
        $defaultItemAgeList = [
            [
                'id'  => 1,
                'value' => 'new',
                'name' => 'New (0 years)',
            ],
            [
                'id'  => 2,
                'value' => 1,
                'name' => 'Nearly New (< 1 year)',
            ],
            [
                'id'  => 3,
                'value' => 3,
                'name' => '1–3 Years',
            ],
            [
                'id'  => 4,
                'value' => 5,
                'name' => '3–5 Years',
            ],
            [
                'id'  => 5,
                'value' => 10,
                'name' => '5–10 Years',
            ],

            [
                'id'  => 6,
                'value' => 20,
                'name' => '10-20 Years',
            ],
            [
                'id'  => 7,
                'value' => '20+',
                'name' => '20+ Years',
            ],
            [
                'id'  => 8,
                'value' => 'vintage',
                'name' => 'Vintage',
            ],
            [
                'id'  => 9,
                'value' => 'antique',
                'name' => 'Antique',
            ],
        ];
        $settingsItemAgeList = get_general_setting('classified_item_age_list', $defaultItemAgeList);
        $this->itemAgeList = collect($settingsItemAgeList);
    }

    public function getDatePostedList()
    {
        $defaultDatePostedList = [
            [
                'id'  => 1,
                'name' => 'Today',
            ],
            [
                'id'  => 3,
                'name' => 'Last 3 Days',
            ],
            [
                'id'  => 7,
                'name' => 'Last 7 Days',
            ],
            [
                'id'  => 30,
                'name' => 'Last 30 Days',
            ],

        ];
        $settingsDatePostedList = get_general_setting('classified_date_posted_list', $defaultDatePostedList);
        $this->datePostedList = collect($settingsDatePostedList);
    }

    public function getMainCategoryList()
    {
        $this->mainCategoryList = $this->classifiedCategoryService->getAll()->get();
    }

    public function getSubCategoryList()
    {
        // Get all possible child categories for selected parent categories
        $allChildCategories = $this->fetchChildrenForCategories($this->mainCategory, 'mainCategory');

        // Filter selected subcategories if we have any previous subcategories
        if ($this->subCategoryList instanceof Collection && $this->subCategoryList->isNotEmpty()) {
            // Filter selected subcategories to only include those with parents still selected
            $this->subCategory = $this->filterSelectedChildren($this->subCategory, $allChildCategories);
        }

        // Update the subcategories list and sort by name
        $this->subCategoryList = $allChildCategories->sortBy('name');
    }

    public function getSellerTypeList()
    {
        $this->sellerTypeList = UserProfileType::cases();
    }

    public function getDeliveryList()
    {
        $this->deliveryList = $this->classifiedDeliveryService->getAll()->get();
    }


    public function getFlashFilterList()
    {
        $defaultFlashFilterList = [
            [
                'id' => 1,
                'name' => 'Ends in < 24 hours',
            ],
            [
                'id' => 2,
                'name' => '1–3 Days Left',
            ],
            [
                'id' => 3,
                'name' => '3–7 Days Left',
            ],
            [
                'id' => 4,
                'name' => '5–10 Years',
            ],
            [
                'id' => 5,
                'name' => 'Price on Request',
            ],
            [
                'id' => 6,
                'name' => 'Negotiable',
            ],
            [
                'id' => 7,
                'name' => 'Best Offer',
            ],
            [
                'id' => 8,
                'name' => 'Free',
            ],
            [
                'id' => 9,
                'name' => 'Auction',
            ],
            [
                'id' => 10,
                'name' => 'Financing Available',
            ],
            [
                'id' => 11,
                'name' => 'Daily/Weekly/Monthly Rate',
            ],
            [
                'id' => 12,
                'name' => 'Price Reduced',
            ],
        ];

        $settingflashFilterList = get_general_setting('classified_flash_filter_list', $defaultFlashFilterList);
        $this->flashFilterList = collect($settingflashFilterList);
    }
    /**
     * Toggle section visibility
     */
    public function toggle(string $section): void
    {
        $prop = 'open' . Str::studly($section);
        if (property_exists($this, $prop)) {
            $this->$prop = ! $this->$prop;
        }
    }

    public function render()
    {
        return view('livewire.frontend.classified.partials.search-classified-sidebar-v1');
    }
}
