<?php

namespace App\Livewire\Frontend\Job;

use Livewire\Component;
use App\Services\Job\JobService;
use App\Services\Job\JobApplicationService;
use App\Services\User\UserService;
use Livewire\WithFileUploads;
use App\Services\Location\CountryService;
use App\Services\Job\JobTypeService;
use Illuminate\Support\Facades\DB;
use App\Enums\JobApplicationStatus;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use Spatie\MediaLibrary\MediaCollections\Models\Media;

class JobApply extends Component
{
    use WithFileUploads;

    protected $jobService;
    protected $jobApplicationService;
    protected $userService;
    protected $countryService;
    protected $jobTypeService;

    public $job;
    public $breadcrumbs;
    public $formConfig = [];
    public $hasAlreadyApplied = false;

    // Dynamic form fields based on config
    public $availability;
    public $availability_date;
    public $video_platform;
    public $youtube_url;
    public $loom_url;
    public $legal_authorization;
    public $visa_sponsorship;
    public $resume;
    public $relevant_experience;
    public $additional_info_1;
    public $additional_info_2;

    // User registration fields (for non-authenticated users)
    public $includePhoto = false;
    public $photo;
    public $country;
    public $city;
    public $firstName;
    public $lastName;
    public $email;
    public $phone;
    public $password;
    public $confirmPassword;
    public $emailUpdates = false;

    public $countryList = [];
    public $jobTypesList = [];
    public $locationData = [];


    protected function messages()
    {
        $configMessages = config('job-application-form.validation_messages', []);

        $userMessages = [
            'firstName.required' => 'First name is required',
            'firstName.min' => 'First name must be at least 2 characters',
            'lastName.required' => 'Last name is required',
            'lastName.min' => 'Last name must be at least 2 characters',
            'email.required' => 'Email is required',
            'email.email' => 'Please enter a valid email address',
            'email.unique' => 'This email is already registered',
            'phone.required' => 'Phone number is required',
            'phone.min' => 'Phone number must be at least 10 digits',
            'password.required' => 'Password is required',
            'password.min' => 'Password must be at least 8 characters',
            'password.regex' => 'Password must contain uppercase, lowercase, number and special character',
            'confirmPassword.required' => 'Please confirm your password',
            'confirmPassword.same' => 'Passwords do not match',
            'country.required' => 'Please select a country',
            'city.required' => 'Please enter a city',
            'photo.image' => 'Photo must be an image file',
            'photo.mimes' => 'Photo must be a JPEG, PNG, or JPG file',
            'photo.max' => 'Photo file size must not exceed 2MB',
        ];

        return array_merge($configMessages, $userMessages);
    }

    public function boot(
        JobService $jobService,
        JobApplicationService $jobApplicationService,
        UserService $userService,
        CountryService $countryService,
        JobTypeService $jobTypeService
    ) {
        $this->jobService = $jobService;
        $this->jobApplicationService = $jobApplicationService;
        $this->userService = $userService;
        $this->countryService = $countryService;
        $this->jobTypeService = $jobTypeService;
    }

    public function mount($slug, $breadcrumbs)
    {
        $this->job = $this->jobService->findBySlug($slug);
        $this->breadcrumbs = $breadcrumbs;

        if (!$this->job) {
            abort(404, 'Job not found');
        }

        // Load form configuration
        $this->formConfig = config('job-application-form.sections', []);

        // Check if user has already applied
        if (Auth::check()) {
            $this->hasAlreadyApplied = $this->job->isApplied();
        }

        // get country list
        $this->getCountryList();
        // get job types
        $this->getJobTypes();
    }

    public function updated($property, $value) {}
    public function getCountryList()
    {
        $this->countryList = $this->countryService->getAll()->get();
    }

    public function getJobTypes()
    {
        $this->jobTypesList = $this->jobTypeService->getAll()->get();
    }


    public function applyJob()
    {

        // Check if user has already applied
        if (Auth::check() && $this->job->isApplied()) {
            session()->flash('error', 'You have already applied to this job.');
            return;
        }

        // Build validation rules from config
        $rules = $this->buildValidationRules();

        // Add user registration fields only if user is not authenticated
        if (!Auth::check()) {
            $rules = array_merge($rules, [
                'firstName' => 'required|min:2|max:30',
                'lastName' => 'required|min:2|max:30',
                'email' => 'required|email|max:30|unique:users,email',
                'phone' => 'required|string|min:10|max:20',
                'password' => 'required|min:8|regex:/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)(?=.*[@$!%*?&])[A-Za-z\d@$!%*?&]/',
                'confirmPassword' => 'required|same:password',
                'country' => 'required',
                'city' => 'required',
                'photo' => 'required_if:includePhoto,true',
            ]);
        }

        $validatedData = $this->validate($rules);

        try {
            DB::beginTransaction();

            if (Auth::user()) {
                $user = Auth::user();
            } else {
                $userData = [
                    'name' => $this->firstName . '_' . $this->lastName,
                    'email' => $this->email,
                    'password' => Hash::make($this->password),
                ];

                $user = $this->userService->create($userData);
                if ($user) {
                    $user->assignRole(ESP_ROLE_SUBSCRIBER);
                    $user->profile()->updateOrCreate([
                        'first_name' => $this->firstName,
                        'last_name' => $this->lastName,
                        'phone' => $this->phone,
                    ]);
                }
            }

            if ($user) {
                // Create job application
                $jobApplication = $this->jobApplicationService->create([
                    'applicant_id' => $user->id,
                    'job_id' => $this->job->id,
                    'application_status' => JobApplicationStatus::Pending,
                    'application_data' => $this->prepareApplicationData(),
                ]);

                // Handle file uploads
                $this->handleFileUploads($jobApplication);

                // upload photo if provided and user is not authenticated
                if (!empty($this->photo) && !Auth::check()) {
                    $user->addMedia($this->photo)->toMediaCollection('avatar');
                }
            }

            DB::commit();
            session()->flash('success', 'Job application submitted successfully');
            $this->hasAlreadyApplied = true;
            $this->dispatch('openModal', component: 'frontend.job.modal.application-confirm', arguments: ['job' => $this->job->id]);
        } catch (\Exception $e) {
            DB::rollBack();
            session()->flash('error', 'Job application failed: ' . $e->getMessage());
            throw $e;
        }
    }

    protected function buildValidationRules()
    {
        $rules = [];

        foreach ($this->formConfig as $sectionKey => $section) {
            if (isset($section['validation'])) {
                if (is_array($section['validation'])) {
                    // Handle multiple validation rules (like video_introduction)
                    foreach ($section['validation'] as $field => $rule) {
                        $rules[$field] = $rule;
                    }
                } else {
                    // Handle single validation rule
                    $rules[$sectionKey] = $section['validation'];
                }
            }

            // Handle conditional fields
            if (isset($section['conditional_field'])) {
                $conditionalField = $section['conditional_field'];
                $rules[$conditionalField['name']] = $conditionalField['validation'];
            }
        }

        return $rules;
    }

    protected function prepareApplicationData()
    {
        $applicationData = [];

        foreach ($this->formConfig as $sectionKey => $section) {
            switch ($section['type']) {
                case 'radio_group':
                case 'radio':
                    if (isset($this->{$sectionKey})) {
                        $applicationData[$sectionKey] = $this->{$sectionKey};
                    }
                    break;

                case 'platform_selector':
                    if (isset($this->video_platform)) {
                        $applicationData['video_platform'] = $this->video_platform;
                        if ($this->video_platform === 'youtube' && $this->youtube_url) {
                            $applicationData['video_url'] = $this->youtube_url;
                        } elseif ($this->video_platform === 'loom' && $this->loom_url) {
                            $applicationData['video_url'] = $this->loom_url;
                        }
                    }
                    break;

                case 'rich_text':
                    if (isset($this->{$sectionKey})) {
                        $applicationData[$sectionKey] = $this->{$sectionKey};
                    }
                    break;

                case 'file':
                    // File handling is done separately
                    break;
            }
        }

        // Handle conditional fields
        if ($this->availability === 'specific_date' && $this->availability_date) {
            $applicationData['availability_date'] = $this->availability_date;
        }

        // Add user location data if not authenticated
        if (!Auth::check()) {
            $applicationData['user_location'] = [
                'country' => $this->country,
                'city' => $this->city
            ];
        }

        return $applicationData;
    }

    protected function handleFileUploads($jobApplication)
    {
        if ($this->resume) {
            $jobApplication->addMedia($this->resume)
                ->toMediaCollection('resume');
        }
    }


    public function getFormConfig()
    {
        return $this->formConfig;
    }

    public function getLeftColumnSections()
    {
        $layout = config('job-application-form.layout.left_column', []);
        $sections = [];

        foreach ($layout as $sectionKey) {
            if (isset($this->formConfig[$sectionKey])) {
                $sections[$sectionKey] = $this->formConfig[$sectionKey];
            }
        }

        return $sections;
    }

    public function getRightColumnSections()
    {
        $layout = config('job-application-form.layout.right_column', []);
        $sections = [];

        foreach ($layout as $sectionKey) {
            if (isset($this->formConfig[$sectionKey])) {
                $sections[$sectionKey] = $this->formConfig[$sectionKey];
            }
        }

        return $sections;
    }

    public function render()
    {
        return view('livewire.frontend.job.job-apply', [
            'leftColumnSections' => $this->getLeftColumnSections(),
            'rightColumnSections' => $this->getRightColumnSections(),
        ]);
    }
}
