<?php

namespace App\Livewire\Frontend\Job;

use App\Services\Job\JobService;
use App\Services\Location\CityService;
use App\Services\Location\CountryService;
use App\Services\Location\StateService;
use App\Services\Job\JobCareerLevelService;
use App\Services\Job\JobTypeService;
use App\Services\Job\JobRoleService;
use App\Enums\JobShift;
use App\Enums\JobWorkMode;
use Illuminate\Support\Collection;
use Illuminate\Support\Str;
use Livewire\Attributes\On;
use Livewire\Component;
use Livewire\WithPagination;
use Livewire\Attributes\Url;
use Livewire\Attributes\Validate;
use App\Models\Location\Country;
use Livewire\Attributes\Isolate;

#[Isolate]
class JobList extends Component
{
    use WithPagination;

    public $perPage = 17;
    public $filters = null;
    public $title   = '';
    public $searchForTxt;

    protected $jobService;
    protected $countryService;
    protected $stateService;
    protected $cityService;
    protected $jobCareerLevelService;
    protected $jobTypeService;
    protected $jobRoleService;
    public $searchParams = [];

    public $openLocation = false;
    public $openCategory = false;
    public $openExperience = false;
    public $openjobType = false;
    public $openJobRole = false;
    public $openWorkingHours = false;

    public $openRemoteWork = false;
    public $openSalary = false;
    public $opend = false;
    public $openDeadline = false;
    public $openFilter = false;
    public $openDatePosted = false;

    public Collection $countryList;
    public Collection $stateList;
    public Collection $cityList;
    public Collection $jobCareerLevelList;
    public Collection $jobTypeList;
    public Collection $jobRoleList;
    public $jobShiftList;

    public $workModeList;

    #[Url(as: 'q')]
    public $search;
    #[Url]
    public $country = '';
    #[Url]
    public $state = '';
    #[Url]
    public $city = '';
    #[Url]
    public $career_level = [];
    #[Url]
    public $job_type = [];
    #[Url]
    public $shift = [];

    #[Url]
    #[Validate('nullable|numeric|min:0|max:999999')]
    public $min_salary;
    #[Url]
    #[Validate('nullable|numeric|min:0|max:999999')]
    public $max_salary;
    #[Url]
    public $date_posted;
    #[Url]
    public $job_role = [];
    #[Url]
    public $openJob = false;
    #[Url]
    public $work_mode = [];
    #[Url]
    public $openJobsOnly = false;
    #[Url]
    public $sortOrder;

    #[Url]
    public $sort = 'created-desc';
    #[Url]
    public $sortBy = 'created-desc';
    #[Url]
    public $order;

    public $page;
    public $breadcrumbs;
    public $countryCode;
    public $sortText;
    public $totalRecords;
    public $totalPages;
    public $currentPage;
    public $datePostedList;

    // Latitude and longitude properties for location search
    public $lat;
    public $lng;
    public $location = '';
    public $radius = 10;
    #[Validate('numeric|min:1|max:500')]
    public $customRadius;

    public function boot(
        JobService $jobService,
        CountryService $countryService,
        StateService $stateService,
        CityService $cityService,
        JobCareerLevelService $jobCareerLevelService,
        JobTypeService $jobTypeService,
        JobRoleService $jobRoleService
    ) {
        $this->jobService = $jobService;
        $this->countryService = $countryService;
        $this->stateService = $stateService;
        $this->cityService = $cityService;
        $this->jobCareerLevelService = $jobCareerLevelService;
        $this->jobTypeService = $jobTypeService;
        $this->jobRoleService = $jobRoleService;
    }

    public function mount(
        $page,
        $breadcrumbs
    ) {
        $this->page = $page;
        $this->breadcrumbs = $breadcrumbs;
        $this->getCountryList();
        $this->getStatesList();
        $this->getCitiesList();
        $this->getJobCareerLevelList();
        $this->getJobTypeList();
        $this->getJobRoleList();
        $this->jobShiftList = JobShift::cases();
        $this->workModeList = JobWorkMode::cases();

        // if (empty($this->date_posted)) {
        //     $this->date_posted = '';
        // }
        $this->getDatePostedList();

        // Set initial sort order
        $this->parseSortOrder();
    }

    public function getDatePostedList()
    {
        $defaultDatePostedList = [
            [
                'id'  => 1,
                'name' => 'Today',
            ],
            [
                'id'  => 3,
                'name' => 'Last 3 Days',
            ],
            [
                'id'  => 7,
                'name' => 'Last 7 Days',
            ],
            [
                'id'  => 30,
                'name' => 'Last 30 Days',
            ],

        ];
        $settingsDatePostedList = get_general_setting('job_date_posted_list', $defaultDatePostedList);
        $this->datePostedList = collect($settingsDatePostedList);
    }



    public function applySearch()
    {
        $this->validate([
            'min_salary' => 'nullable|numeric|min:0|max:999999',
            'max_salary' => [
                'nullable',
                'numeric',
                'min:0',
                'max:999999',
                function ($attribute, $value, $fail) {
                    if ($value && $this->min_salary && $value < $this->min_salary) {
                        $fail('Maximum salary cannot be less than minimum salary.');
                    }
                },
            ],
        ]);
        $this->openFilter = false;
        // Build search parameters
        $searchParams = [
            'search' => $this->search,
            'city' => $this->city,
            'state' => $this->state,
            'country' => $this->country,
            'date_posted' => $this->date_posted,
            'job_role' => $this->job_role,
            'job_type' => $this->job_type,
            'min_salary' => $this->min_salary,
            'max_salary' => $this->max_salary,
            'career_level' => $this->career_level,
            'shift' => $this->shift,
            'work_mode' => $this->work_mode,
            'open_jobs_only' => $this->openJobsOnly,
            'sort' => $this->sort,
        ];

        // Generate search text
        $this->generateSearchForText();
    }

    public function generateSearchForText()
    {
        $searchText = '';
        $location = '';

        if (!empty($this->search)) {
            $searchText .= $this->search;
        }

        if (!empty($this->career_level)) {
            $careerLevelList = [];
            foreach ($this->career_level as $key => $careerLevel) {
                $careerLevelData = $this->jobCareerLevelList->where('slug', $careerLevel)->first();
                if ($careerLevelData) {
                    $careerLevelList[] = $careerLevelData->career_level_name;
                }
            }
            if (!empty($careerLevelList)) {
                $searchText .= ' ' . $careerLevelList[0];
                if (count($this->career_level) > 1) {
                    $searchText .= ' +' . (count($this->career_level) - 1);
                }
            }
        }

        if (!empty($this->job_type)) {
            $typeList = [];
            foreach ($this->job_type as $key => $type) {
                if ($key === 0) {
                    $typeData = $this->jobTypeList->where('slug', $type)->first();
                    if ($typeData) {
                        $typeList[] = $typeData->job_type_name;
                    }
                    break;
                }
            }
            if (!empty($typeList)) {
                $searchText .= ' ' . $typeList[0];
                if (count($this->job_type) > 1) {
                    $searchText .= ' +' . (count($this->job_type) - 1);
                }
            }
        }

        if (!empty($this->city)) {
            $location .= ' ' . $this->city;
        } else {
            if (!empty($this->state)) {
                $stateData = $this->stateList->where('slug', $this->state)->first();
                if ($stateData) {
                    $location .= ' ' . $stateData->state_name;
                }
            }
            if (!empty($this->country)) {
                $countryData = $this->countryList->where('slug', $this->country)->first();
                if ($countryData) {
                    $location .= ' ' . $countryData->country_name;
                }
            }
        }

        if (!empty($location)) {
            $searchText .= ' near' . $location;
        }

        if (!empty($this->min_salary) || !empty($this->max_salary)) {
            $searchText .= ' salary: ';
            if (!empty($this->min_salary)) {
                $searchText .= '$' . $this->min_salary;
            }
            $searchText .= ' - ';
            if (!empty($this->max_salary)) {
                $searchText .= '$' . $this->max_salary;
            }
        }

        if (!empty($this->shift)) {
            $shiftList = [];
            foreach ($this->shift as $key => $shift) {
                if ($key === 0) {
                    $shiftData = collect($this->jobShiftList)->where('value', $shift)->first();
                    if ($shiftData) {
                        $shiftList[] = $shiftData->name;
                    }
                    break;
                }
            }
            if (!empty($shiftList)) {
                $searchText .= ' ' . $shiftList[0];
                if (count($this->shift) > 1) {
                    $searchText .= ' +' . (count($this->shift) - 1);
                }
            }
        }

        if (!empty($this->work_mode)) {
            $workModeList = [];
            foreach ($this->work_mode as $key => $mode) {
                if ($key === 0) {
                    $modeData = collect($this->workModeList)->where('value', $mode)->first();
                    if ($modeData) {
                        $workModeList[] = $modeData->name;
                    }
                    break;
                }
            }
            if (!empty($workModeList)) {
                $searchText .= ' ' . $workModeList[0];
                if (count($this->work_mode) > 1) {
                    $searchText .= ' +' . (count($this->work_mode) - 1);
                }
            }
        }

        $this->searchForTxt = $searchText;
    }

    /**
     * Parse location string into lat/lng coordinates
     */
    private function parseLocationString()
    {
        if (!empty($this->location)) {
            $splitLocation = explode(',', $this->location);
            if (count($splitLocation) >= 2) {
                $this->lat = trim($splitLocation[0]);
                $this->lng = trim($splitLocation[1]);
            }
        }
    }

    public function updated($property, $value)
    {
        if ($property == 'country') {
            if ($value != 'all' && $value != '') {
                $this->getStatesList();
            }
            $this->reset(['state', 'city']);
            $this->applySearch();
        }

        if ($property == 'state') {
            $this->getCitiesList();
            $this->reset(['city']);
            $this->applySearch();
        }

        if ($property == 'city') {
            if (empty($value)) {
                $this->lat = null;
                $this->lng = null;
                $this->location = '';
            }
            $this->applySearch();
        }

        // Handle updates to lat or lng
        if ($property === 'lat' || $property === 'lng') {
            // Update location string when coordinates change
            if (!empty($this->lat) && !empty($this->lng)) {
                $this->location = $this->lat . ',' . $this->lng;
            } elseif (empty($this->lat) || empty($this->lng)) {
                $this->location = '';
            }
        }

        // Handle direct updates to location string
        if ($property === 'location') {
            $this->parseLocationString();
            $this->applySearch();
        }

        if ($property === 'radius') {
            if ($this->radius != 'custom') {
                $this->applySearch();
            }
        }

        if ($property === 'customRadius') {
            $this->applySearch();
        }

        if ($property == 'date_posted') {
            $this->applySearch();
        }

        if (Str::startsWith($property, 'job_role.')) {
            $this->applySearch();
        }

        if (Str::startsWith($property, 'job_type.')) {
            $this->applySearch();
        }

        if (Str::startsWith($property, 'career_level.')) {
            $this->applySearch();
        }

        if (Str::startsWith($property, 'shift.')) {
            $this->applySearch();
        }

        if (Str::startsWith($property, 'work_mode.')) {
            $this->applySearch();
        }

        if ($property == 'open_jobs_only') {
            $this->applySearch();
        }

        if ($property == 'min_salary' || $property == 'max_salary') {
            $this->applySearch();
        }

        if ($property === 'country') {
            if ($value != 'all' || $value != '') {
                $this->getStatesList();
                $countryModel = Country::where('slug', $value)->first();
                if ($countryModel) {
                    $this->countryCode = $countryModel->country_code;
                }
            } else {
                $this->countryCode = '';
            }
            // When country changes, reset state and city
            $this->state = '';
            $this->city = '';
            $this->lat = null;
            $this->lng = null;
        }
    }

    public function getCountryList()
    {
        $this->countryList = $this->countryService->getAll()->get();
    }

    public function resetFilters()
    {
        $this->reset([
            'search',
            'country',
            'state',
            'city',
            'date_posted',
            'min_salary',
            'max_salary',
            'job_role',
            'career_level',
            'shift',
            'work_mode',
            'job_type',
            'lat',
            'lng',
            'location',
            'radius',
            'customRadius'
        ]);

        // Reset dependent lists
        $this->stateList = collect();
        $this->cityList = collect();
        $this->date_posted = '';

        $this->applySearch();
    }

    private function fetchStatesForCountry($countrySlug): Collection
    {
        $allStates = collect();

        if (!empty($countrySlug) && is_string($countrySlug)) {
            $country = $this->countryService->getCountryBySlug($countrySlug);

            if ($country) {
                $states = $this->stateService->getAll(['country_id' => $country->id])->get();
                $allStates = $states instanceof Collection ? $states : collect($states);
            }
        }

        return $allStates->unique('id')->values();
    }

    private function fetchCitiesForState($stateSlug): Collection
    {
        $allCities = collect();

        if (!empty($stateSlug) && is_string($stateSlug)) {
            $state = $this->stateService->getStateBySlug($stateSlug);

            if ($state) {
                $cities = $this->cityService->getAll(['state_id' => $state->id])->get();
                $allCities = $cities instanceof Collection ? $cities : collect($cities);
            }
        }

        return $allCities->unique('id')->values();
    }

    public function getStatesList()
    {
        $stateList = collect();
        if (!empty($this->country)) {
            $stateList = $this->fetchStatesForCountry($this->country);
        }
        $this->stateList = $stateList;
    }

    public function getCitiesList()
    {
        $cityList = collect();

        if (!empty($this->state)) {
            $cityList = $this->fetchCitiesForState($this->state);
        }
        $this->cityList = $cityList;
    }

    public function getJobCareerLevelList()
    {
        $this->jobCareerLevelList = $this->jobCareerLevelService->getAll()->get();
    }

    public function getJobTypeList()
    {
        $this->jobTypeList = $this->jobTypeService->getAll()->get();
    }

    public function getJobRoleList()
    {
        $this->jobRoleList = $this->jobRoleService->getAll()->onlyParent()->get();
    }

    public function applySortOrder()
    {
        $this->parseSortOrder();
        $this->applySearch();
    }

    public function parseSortOrder()
    {
        if ($this->sort == 'created-desc') {
            $this->sortOrder = 'created_at';
            $this->order = 'desc';
            $this->sortText = 'Most Recent';
        } elseif ($this->sort == 'created-asc') {
            $this->sortOrder = 'created_at';
            $this->order = 'asc';
            $this->sortText = 'Oldest First';
        } elseif ($this->sort == 'salary-desc') {
            $this->sortOrder = 'min_salary';
            $this->order = 'desc';
            $this->sortText = 'Salary: High to Low';
        } elseif ($this->sort == 'salary-asc') {
            $this->sortOrder = 'min_salary';
            $this->order = 'asc';
            $this->sortText = 'Salary: Low to High';
        } else {
            $this->sortOrder = 'created_at';
            $this->order = 'desc';
            $this->sortText = 'Best Match';
        }
    }

    public function render()
    {
        $args = [
            'search' => $this->search,
            'city' => $this->city,
            'state' => $this->state,
            'country' => $this->country,
            'job_role' => $this->job_role,
            'job_type' => $this->job_type,
            'min_salary' => $this->min_salary,
            'max_salary' => $this->max_salary,
            'date_posted' => $this->date_posted,
            'shift' => $this->shift,
            'work_mode' => $this->work_mode,
            'open_jobs_only' => $this->openJobsOnly,
            'career_level' => $this->career_level,
            'lat' => $this->lat,
            'lng' => $this->lng,
            'order_by' => $this->sortOrder,
            'order' => $this->order,
        ];

        // Merge searchParams if they exist
        if (!empty($this->searchParams)) {
            $args = array_merge($args, $this->searchParams);
        }

        $jobs = $this->jobService->getJobList($args, $this->perPage);
        return view('livewire.frontend.job.job-list', ['jobs' => $jobs]);
    }
}
