<?php

namespace App\Livewire\Frontend\Job\Partials;

use App\Services\Location\CityService;
use App\Services\Location\CountryService;
use App\Services\Location\StateService;
use Illuminate\Support\Collection;
use Illuminate\Support\Str;
use Livewire\Attributes\Url;
use Livewire\Attributes\Validate;
use Livewire\Component;
use App\Services\Job\JobCareerLevelService;
use App\Services\Job\JobTypeService;
use App\Services\Job\JobRoleService;
use App\Enums\JobShift;
use App\Enums\JobWorkMode;
use Livewire\Attributes\Isolate;

#[Isolate]
class SearchJobSidebarV1 extends Component
{

    protected $countryService;
    protected $stateService;
    protected $cityService;
    protected $jobCareerLevelService;
    protected $jobTypeService;
    protected $jobRoleService;

    public $openLocation = false;
    public $openCategory = false;
    public $openExperience = false;
    public $openjobType = false;
    public $openWorkingHours = false;
    public $openRemoteWork = false;
    public $openSalary = false;
    public $opend = false;
    public $openDeadline = false;
    public $openFilter = false;


    public Collection $countryList;
    public Collection $stateList;
    public Collection $cityList;
    public Collection $jobCareerLevelList;
    public Collection $jobTypeList;
    public Collection $jobRoleList;
    public $jobShiftList;
    public $workModeList;

    #[Url]
    public $search;
    #[Url]
    public $country = '';
    #[Url]
    public $state = '';
    #[Url]
    public $city = '';
    #[Url]
    public $career_level = [];
    #[Url]
    public $job_type = [];
    #[Url]
    public $shift = [];

    #[Url]
    #[Validate('numeric|max:999999')]
    public $min_salary;
    #[Url]
    #[Validate('numeric|max:999999')]
    public $max_salary;
    #[Url]
    public $date_posted;
    #[Url]
    public $job_role = [];
    #[Url]
    public $openJob = false;
    #[Url]
    public $work_mode = [];
    #[Url]
    public $openJobsOnly = false;

    public function boot(CountryService $countryService, StateService $stateService, CityService $cityService, JobCareerLevelService $jobCareerLevelService, JobTypeService $jobTypeService, JobRoleService $jobRoleService)
    {
        $this->countryService = $countryService;
        $this->stateService   = $stateService;
        $this->cityService    = $cityService;
        $this->jobCareerLevelService = $jobCareerLevelService;
        $this->jobTypeService = $jobTypeService;
        $this->jobRoleService = $jobRoleService;
    }

    public function mount()
    {
        $this->getCountryList();

        // Load states if country is selected
        $this->getStatesList();

        // Load cities if states are selected
        $this->getCitiesList();

        $this->getJobCareerLevelList();

        $this->getJobTypeList();

        $this->getJobRoleList();

        $this->jobShiftList = JobShift::cases();

        $this->workModeList = JobWorkMode::cases();

        // Initialize date_posted if empty
        if (empty($this->date_posted)) {
            $this->date_posted = ''; // Start with no dates selected
        }
    }

    public function applySearch()
    {
        $this->validate([
            'min_salary' => 'nullable|numeric|max:999999',
            'max_salary' => [
                'nullable',
                'numeric',
                'max:999999',
                function ($attribute, $value, $fail) {
                    if ($value && $this->min_salary && $value < $this->min_salary) {
                        $fail('Maximum salary cannot be less than minimum salary.');
                    }
                },
            ],
        ]);
        $this->openFilter = false;
        // Build search parameters
        $searchParams = [
            'search' => $this->search,
            'city' => $this->city,
            'state' => $this->state,
            'country' => $this->country,
            'date_posted' => $this->date_posted,
            'job_role' => $this->job_role,
            'job_type' => $this->job_type,
            'min_salary' => $this->min_salary,
            'max_salary' => $this->max_salary,
            'career_level' => $this->career_level,
            'shift' => $this->shift,
            'work_mode' => $this->work_mode,
            'open_jobs_only' => $this->openJobsOnly,
        ];
        // Dispatch event to SearchResultListView component
        $this->dispatch('search-jobs', $searchParams);
    }


    public function updated($property, $value)
    {
        if ($property == 'country') {
            $this->getStatesList();
            $this->reset(['state', 'city']);
            $this->applySearch();
        }

        if ($property == 'state') {
            $this->getCitiesList();
            $this->reset(['city']);
            $this->applySearch();
        }

        if ($property == 'city') {
            $this->applySearch();
        }

        if ($property == 'date_posted') {
            $this->applySearch();
        }

        if (Str::startsWith($property, 'job_role.')) {
            $this->applySearch();
        }

        if (Str::startsWith($property, 'job_type.')) {
            $this->applySearch();
        }

        if (Str::startsWith($property, 'career_level.')) {
            $this->applySearch();
        }

        if (Str::startsWith($property, 'shift.')) {
            $this->applySearch();
        }

        if (Str::startsWith($property, 'work_mode.')) {
            $this->applySearch();
        }

        if ($property == 'open_jobs_only') {
            $this->applySearch();
        }

        if ($property == 'min_salary' || $property == 'max_salary') {
            $this->applySearch();
        }
    }
    public function getCountryList()
    {
        $this->countryList = $this->countryService->getAll()->get();
    }
    /**
     * Reset all filters to their default values
     */
    public function resetFilters()
    {
        $this->reset([
            'search',
            'country',
            'state',
            'city',
            'date_posted',
            'min_salary',
            'max_salary',
            'job_role',
            'career_level',
            'shift',
            'work_mode',
            'open_jobs_only',
            'job_type',
        ]);

        // Reset dependent lists
        $this->stateList   = collect();
        $this->cityList    = collect();
        $this->date_posted = '';

        $this->applySearch();
    }

    /**
     * Get all states for selected country
     *
     * @param string|null $countrySlug Country slug to get states for
     * @return Collection Collection of states
     */
    private function fetchStatesForCountry($countrySlug): Collection
    {
        $allStates = collect();

        // If country is provided
        if (! empty($countrySlug) && is_string($countrySlug)) {
            $country = $this->countryService->getCountryBySlug($countrySlug);

            if ($country) {
                // Get states for this country
                $states = $this->stateService->getAll(['country_id' => $country->id])->get();

                // Ensure states is a collection
                $allStates = $states instanceof Collection ? $states : collect($states);
            }
        }

        // Return states
        return $allStates->unique('id')->values();
    }

    /**
     * Get all cities for selected state
     *
     * @param string|null $stateSlug State slug to get cities for
     * @return Collection Collection of cities
     */
    private function fetchCitiesForState($stateSlug): Collection
    {
        $allCities = collect();

        // If state is provided
        if (! empty($stateSlug) && is_string($stateSlug)) {
            $state = $this->stateService->getStateBySlug($stateSlug);

            if ($state) {
                // Get cities for this state
                $cities = $this->cityService->getAll(['state_id' => $state->id])->get();

                // Ensure cities is a collection
                $allCities = $cities instanceof Collection ? $cities : collect($cities);
            }
        }

        // Return cities
        return $allCities->unique('id')->values();
    }

    public function getStatesList()
    {
        $stateList = collect();
        // If no country is selected, clear states and cities
        if (!empty($this->country)) {
            // Get all states for selected country
            $stateList = $this->fetchStatesForCountry($this->country);
        }
        $this->stateList = $stateList;
    }

    public function getCitiesList()
    {
        $cityList = collect();

        if (!empty($this->state)) {
            // Get all cities for selected state
            $cityList = $this->fetchCitiesForState($this->state);
        }
        $this->cityList = $cityList;
    }

    public function getJobCareerLevelList()
    {
        $this->jobCareerLevelList = $this->jobCareerLevelService->getAll()->get();
    }

    public function getJobTypeList()
    {
        $this->jobTypeList = $this->jobTypeService->getAll()->get();
    }

    public function getJobRoleList()
    {
        $this->jobRoleList = $this->jobRoleService->getAll()->onlyParent()->get();
    }

    public function render()
    {
        return view('livewire.frontend.job.partials.search-job-sidebar-v1');
    }
}
