<?php

namespace App\Livewire\Frontend\Member\Listing\SearchResult;

use Illuminate\Support\Collection;
use Livewire\Attributes\Url;
use Illuminate\Support\Str;

trait MemberCategoryTrait
{
    // Services
    protected $memberCategoryService;

    // Lists for dropdowns
    public $categoriesList;
    public $subcategoriesList;
    public $specificCategoriesList;

    // Collapse states
    public bool $openCategory = true;
    public bool $openSubcategory = true;
    public bool $openSpecificServices = false;

    #[Url]
    public $category = [];

    #[Url]
    public $subcategory = [];

    #[Url]
    public $service = [];

    #[Url]
    public $specificCategory = [];

    public function getParentCategories()
    {
        $this->categoriesList = $this->memberCategoryService->getParentCategories()->sortBy('name');
    }

    /**
     * Helper method to get child categories for a list of parent categories
     *
     * @param  array|Collection  $parentSlugs  Slugs of parent categories
     * @param  string  $parentType  Type of parent ('category' or 'subcategory')
     * @return Collection Collection of child categories
     */
    private function fetchChildrenForCategories($parentSlugs, string $parentType): Collection
    {
        // Create a collection to store child categories
        $allChildren = collect();

        // Convert to collection if not already
        $parentSlugs = collect($parentSlugs);

        // If parent slugs are provided
        if ($parentSlugs->isNotEmpty()) {
            // Process each parent slug
            foreach ($parentSlugs as $slug) {
                $parent = $this->memberCategoryService->getCategoryBySlug($slug);

                if ($parent) {
                    // Get children for this parent
                    $children = $this->memberCategoryService->getChildCategories($parent->id);

                    // Ensure children is a collection
                    $children = $children instanceof Collection ? $children : collect($children);

                    // Merge with main collection
                    $allChildren = $allChildren->merge($children);
                }
            }
        }

        // Return unique children sorted by name
        return $allChildren->unique('id')->values()->sortBy('name');
    }

    /**
     * Filter selected children based on what's available in the full children list
     *
     * @param  array|Collection  $selectedSlugs  Currently selected child slugs
     * @param  Collection  $availableChildren  Available child categories to filter against
     * @return array Filtered selection array
     */
    private function filterSelectedChildren($selectedSlugs, Collection $availableChildren): array
    {
        $childrenIdsToKeep = $availableChildren->pluck('id')->toArray();

        return collect($selectedSlugs)
            ->filter(function ($slug) use ($childrenIdsToKeep) {
                $child = $this->memberCategoryService->getCategoryBySlug($slug);

                return $child && in_array($child->id, $childrenIdsToKeep);
            })
            ->toArray();
    }

    public function getChildCategories()
    {
        // Get all possible child categories for selected parent categories
        $allChildCategories = $this->fetchChildrenForCategories($this->category, 'category');

        // Filter selected subcategories if we have any previous subcategories
        if ($this->subcategoriesList instanceof Collection && $this->subcategoriesList->isNotEmpty()) {
            // Filter selected subcategories to only include those with parents still selected
            $this->subcategory = $this->filterSelectedChildren($this->subcategory, $allChildCategories);

            // If all subcategories were removed, clear related data
            if (empty($this->subcategory)) {
                $this->service                = [];
                $this->specificCategoriesList = collect();
            } else {
                // Update specific categories based on remaining subcategories
                $this->getSpecificCategories();
            }
        }

        // Update the subcategories list and sort by name
        $this->subcategoriesList = $allChildCategories->sortBy('name');
    }

    public function getSpecificCategories()
    {
        // Get all possible specific categories for selected subcategories
        $allSpecificCategories = $this->fetchChildrenForCategories($this->subcategory, 'subcategory');

        // Filter selected services if needed
        if (! empty($this->service) && $this->specificCategoriesList instanceof Collection && $this->specificCategoriesList->isNotEmpty()) {
            // Filter selected services to only include those with parents still selected
            $this->service = $this->filterSelectedChildren($this->service, $allSpecificCategories);
        }

        // Update the specific categories list and sort by name
        $this->specificCategoriesList = $allSpecificCategories->sortBy('name');
    }

    public function updatedCategory($value): void
    {
        if (! empty($value)) {
            $this->openSubcategory = true;
        }
    }

    /**
     * Generate category-related search text
     */
    protected function generateCategorySearchText(): string
    {
        $searchText = '';

        if (! empty($this->category)) {
            $categoryList = [];
            foreach ((array) $this->category as $key => $category) {
                $categoryData = $this->categoriesList->where('slug', $category)->first();
                if ($categoryData && $key < 2) {
                    $categoryList[] = $categoryData->category_name;
                }
            }

            if (! empty($categoryList)) {
                $searchText .= ' for ' . implode(', ', $categoryList);
                if (is_array($this->category) && count($this->category) > 2) {
                    $searchText .= ' and ' . (count($this->category) - 2) . ' more';
                }
            }
        }

        return Str::headline($searchText);
    }

    /**
     * Reset category-related filters
     */
    protected function resetCategoryFilters()
    {
        $this->reset([
            'category',
            'subcategory',
            'service',
            'specificCategory',
        ]);

        // Reset dependent lists
        $this->subcategoriesList      = collect();
        $this->specificCategoriesList = collect();
    }
}