<?php

namespace App\Livewire\Frontend\Member\Listing\SearchResult;

use App\Services\User\MemberCategoryService;
use App\Services\User\UserService;
use Livewire\Component;
use App\Services\Location\CityService;
use Illuminate\Support\Str;
use Livewire\Attributes\Url;
use Livewire\WithPagination;
use App\Enums\MembershipType;

class MemberCityGrid extends Component
{
    use MemberCategoryTrait, WithPagination;

    protected $cityService;
    protected $userService;
    protected $profileMetaCount = [];
    protected $memberList = [];
    public $areas;

    // Search properties
    #[Url(as: 'q')]
    public $search;

    // Sorting properties
    #[Url]
    public $sort = 'created-desc';
    public $sortOrder;
    public $order;
    public $sortText;
    public $state;
    public $country;
    public $city;

    // Pagination properties
    public $perPage = 15;
    public $totalRecords;
    public $currentPage;
    public $totalPages;


    public $profileFeatures;

    // Rating properties
    #[Url]
    public $rating = [];

    #[Url]
    public $features = [];

    public $profileReviewsCount;
    public $profileEmergencyCount;
    public $profileOpenNowCount;

    // Search text display
    public $searchForTxt;

    // Country code for view
    public $countryCode;

    public function boot(UserService $userService, MemberCategoryService $memberCategoryService)
    {
        $this->userService = $userService;
        $this->memberCategoryService = $memberCategoryService;
    }

    public function mount(CityService $cityService, $citySlug)
    {
        $this->cityService = $cityService;
        $city = $this->cityService->findBySlug($citySlug);
        $this->city = $city->slug;
        $this->state = $city->state?->slug;
        $this->country = $city->state?->country?->slug;
        $this->areas = $city->areas;
        $this->categoriesList = collect();
        $this->subcategoriesList = collect();
        $this->specificCategoriesList = collect();

        $this->getParentCategories();
        $this->getProfileFeatures();
        $this->getRatingData();

        $this->generateSearchForText();

        $this->parseSortOrder();
        $this->memberList = $this->getAllMembers();

        $this->getProfileMetaCount();
    }

    public function updated($property, $value)
    {
        // Handle category updates
        if (Str::startsWith($property, 'category.')) {
            $this->getChildCategories();
            if (empty($this->category)) {
                $this->subcategory            = [];
                $this->service                = [];
                $this->subcategoriesList      = collect();
                $this->specificCategoriesList = collect();
            }
            // Ensure categories are sorted
            $this->categoriesList = $this->categoriesList->sortBy('name');
        }

        if (Str::startsWith($property, 'subcategory.')) {
            $this->getSpecificCategories();
            if (empty($this->subcategory)) {
                $this->specificCategoriesList = collect();
                $this->service                = [];
            }
            // Ensure subcategories are sorted
            $this->subcategoriesList = $this->subcategoriesList->sortBy('name');
        }
    }

    /**
     * Reset all filters to their default values
     */
    public function resetFilters()
    {
        $this->reset([
            'search',
            'category',
            'subcategory',
            'specificCategory',
        ]);

        // Reset dependent lists
        $this->subcategoriesList      = collect();
        $this->specificCategoriesList = collect();
    }

    public function parseSortOrder()
    {
        if ($this->sort == 'created-desc') {
            $this->sortOrder = 'created_at';
            $this->order     = 'desc';
            $this->sortText  = 'Most Recent';
        } elseif ($this->sort == 'created-asc') {
            $this->sortOrder = 'created_at';
            $this->order     = 'asc';
            $this->sortText  = 'Most Recent';
        } elseif ($this->sort == 'relevent') {
            $this->sortOrder = 'created_at';
            $this->order     = 'desc';
            $this->sortText  = 'Most Relevant';
        } elseif ($this->sort == 'rating-desc') {
            $this->sortOrder = 'profile_rating';
            $this->order     = 'desc';
            $this->sortText  = 'Highest Rated';
        } elseif ($this->sort == 'rating-asc') {
            $this->sortOrder = 'profile_rating';
            $this->order     = 'asc';
            $this->sortText  = 'Lowest Rated';
        } elseif ($this->sort == 'featured-first') {
            $this->sortOrder = 'featured_first';
            $this->order     = 'desc';
            $this->sortText  = 'Featured First';
        } else {
            $this->sortOrder = 'created_at';
            $this->order     = 'desc';
            $this->sortText  = 'Best Match';
        }
    }

    public function applySortOrder()
    {
        $this->parseSortOrder();
        $this->resetPage();
    }

    public function generateSearchForText()
    {
        $searchText = '';

        if (! empty($this->search)) {
            $searchText .= $this->search;
        }

        // Add category text using the trait method
        $searchText .= $this->generateCategorySearchText();

        $this->searchForTxt = $searchText;
    }

    public function applySearch()
    {
        $this->generateSearchForText();
        $this->resetPage();
    }

    /**
     * Get rating data with real counts from UserService
     */
    public function getRatingData(): array
    {
        // Get real counts from UserService
        $ratingCounts = $this->userService->getUserCountsByRating();

        return [
            ['value' => 1, 'label' => '1 star', 'count' => $ratingCounts[1] ?? 0],
            ['value' => 2, 'label' => '2 stars', 'count' => $ratingCounts[2] ?? 0],
            ['value' => 3, 'label' => '3 stars', 'count' => $ratingCounts[3] ?? 0],
            ['value' => 4, 'label' => '4 stars', 'count' => $ratingCounts[4] ?? 0],
            ['value' => 5, 'label' => '5 stars', 'count' => $ratingCounts[5] ?? 0],
        ];
    }

    public function getProfileFeatures()
    {
        $features = [
            ['id' => 'reviews', 'name' => 'With Customer Reviews'],
            ['id' => 'emergency', 'name' => 'Emergency (24/7)'],
            ['id' => 'open_now', 'name' => 'Open Now'],
        ];
        $profileFeatures       = get_general_setting('profile_features', $features);
        $this->profileFeatures = collect($profileFeatures);
    }

    // get profile meta counts
    public function getProfileMetaCount()
    {
        // Get member IDs from the member list
        // Get profile meta counts for the specific members
        $this->profileReviewsCount = esp_get_profile_meta_count('reviews');
        $this->profileEmergencyCount = esp_get_profile_meta_count('emergency');
        $this->profileOpenNowCount = esp_get_profile_meta_count('open_now');
    }

    public function getAllMembers()
    {
        $args = [
            'role' => ESP_ROLE_MEMBER,
            'search' => $this->search,
            'category' => $this->category,
            'subcategory' => $this->subcategory,
            'service' => $this->service,
            'specificCategory' => $this->specificCategory,
            'city' => $this->city,
            'state' => $this->state,
            'country' => $this->country,
            'limit' => $this->perPage,
            'order_by' => $this->sortOrder,
            'order' => $this->order,
        ];
        return $this->userService->getMemberList($args);
    }


    public function render()
    {

        // Get all members
        $members = $this->getAllMembers();
        
        // $this->memberList = $members;

        // Split the collection into featured and non-featured
        $featuredMembers = $members->getCollection()->filter(function ($member) {
            return $member->membershipType === MembershipType::Featured;
        })->take(1);


        $this->totalRecords = $members->total();
        $this->totalPages = $members->lastPage();
        $this->currentPage = $members->currentPage();

        return view('livewire.frontend.member.listing.search-result.member-city-grid', compact('members', 'featuredMembers'));
    }
}
