<?php

namespace App\Livewire\Frontend\Member\Listing\SearchResult;

use App\Enums\MembershipType;
use App\Models\Location\Country;
use App\Services\User\UserService;
use Illuminate\Support\Collection;
use Illuminate\Support\Str;
use Livewire\Attributes\On;
use Livewire\Attributes\Url;
use Livewire\Attributes\Validate;
use Livewire\WithPagination;

trait MemberSearchTrait
{
    use WithPagination;

    // Services
    protected $userService;

    protected $countryService;

    protected $stateService;

    protected $cityService;

    protected $reviewService;

    // properties
    public $page;

    public $viewType;

    public $searchParams = [];

    public $searchForTxt;

    // Lists for dropdowns
    public $countryList = [];

    public $stateList;

    public $cityList;

    public $radiusList;

    public $profileFeatures;

    public $countryCode;

    // Collapse states
    public bool $openRating = false;

    public bool $openLocation = false;

    public bool $openFeatures = false;

    #[Validate('numeric|min:1|max:500')]
    public $customRadius;

    #[Url(as: 'q')]
    public $search;

    #[Url]
    public $country = '';

    #[Url]
    public $state = '';

    #[Url]
    public $city = '';

    #[Url]
    public $radius = 10;

    #[Url]
    public $rating = [];

    #[Url]
    public $features = [];

    #[Url]
    public $location = '';

    #[Url]
    public $membershipType = MembershipType::Free->value;

    #[Url]
    public $sort = 'created-desc';

    // Latitude and longitude properties for location search
    public $lat;

    public $lng;

    public $sortOrder;

    public $order;

    public $sortText;

    public $perPage = 15;

    public $totalRecords;

    public $totalPages;

    public $currentPage;

    protected $members;

    public $reviewsStats;

    public $breadcrumbs;

    public $profileReviewsCount;
    public $profileEmergencyCount;
    public $profileOpenNowCount;

    public function updated($property, $value)
    {
        // Handle category updates if using MemberCategoryTrait
        if (method_exists($this, 'getChildCategories') && Str::startsWith($property, 'category.')) {
            $this->getChildCategories();
            if (empty($this->category)) {
                $this->subcategory            = [];
                $this->service                = [];
                $this->subcategoriesList      = collect();
                $this->specificCategoriesList = collect();
            }
            // Ensure categories are sorted
            if (isset($this->categoriesList)) {
                $this->categoriesList = $this->categoriesList->sortBy('name');
            }
            $this->applySearch();
        }

        if (method_exists($this, 'getSpecificCategories') && Str::startsWith($property, 'subcategory.')) {
            $this->getSpecificCategories();
            if (empty($this->subcategory)) {
                $this->specificCategoriesList = collect();
                $this->service                = [];
            }
            // Ensure subcategories are sorted
            if (isset($this->subcategoriesList)) {
                $this->subcategoriesList = $this->subcategoriesList->sortBy('name');
            }
            $this->applySearch();
        }

        if (method_exists($this, 'getSpecificCategories') && Str::startsWith($property, 'service.')) {
            $this->getSpecificCategories();
            if (empty($this->service)) {
                $this->specificCategoriesList = collect();
                $this->service                = [];
            }
            // Ensure specific categories are sorted
            if (isset($this->specificCategoriesList)) {
                $this->specificCategoriesList = $this->specificCategoriesList->sortBy('name');
            }
            $this->applySearch();
        }

        if ($property === 'country') {
            if ($value != 'all') {
                $this->getStatesList();
                $countryModel = Country::where('slug', $value)->first();
                if ($countryModel) {
                    $this->countryCode = $countryModel->country_code;
                }
            } else {
                $this->countryCode = '';
            }
            // When country changes, reset state and city
            $this->state        = '';
            $this->city         = '';
            $this->lat          = null;
            $this->lng          = null;
            $this->location     = '';
            $this->radius       = 10;
            $this->customRadius = null;
            $this->applySearch();
        }

        if ($property === 'state') {
            $this->getCitiesList();
            if (empty($this->state)) {
                $this->city = '';
            }
            $this->applySearch();
        }

        if ($property === 'city') {
            if (empty($value)) {
                $this->lat      = null;
                $this->lng      = null;
                $this->location = '';
            }
            $this->applySearch();
        }

        // Handle updates to lat or lng
        if ($property === 'lat' || $property === 'lng') {
            // Update location string when coordinates change
            if (! empty($this->lat) && ! empty($this->lng)) {
                $this->location = $this->lat . ',' . $this->lng;
            } elseif (empty($this->lat) || empty($this->lng)) {
                $this->location = '';
            }
        }

        // Handle direct updates to location string
        if ($property === 'location') {
            $this->parseLocationString();
            $this->applySearch();
        }

        if ($property === 'radius') {
            if ($this->radius != 'custom') {
                $this->applySearch();
            }
        }

        if (Str::startsWith($property, 'rating.')) {
            $this->applySearch();
        }

        if (Str::startsWith($property, 'features.')) {
            $this->applySearch();
        }

        if ($property === 'customRadius') {
            $this->applySearch();
        }
    }

    /**
     * Reset all filters to their default values
     */
    public function resetFilters()
    {
        $resetFields = [
            'search',
            'country',
            'state',
            'city',
            'radius',
            'rating',
            'customRadius',
            'features',
            'lat',
            'lng',
            'location',
            'features',
        ];

        // Add category fields if using MemberCategoryTrait
        if (method_exists($this, 'resetCategoryFilters')) {
            $resetFields = array_merge($resetFields, [
                'category',
                'subcategory',
                'service',
                'specificCategory',
            ]);
        }

        $this->reset($resetFields);

        // Reset dependent lists
        $this->stateList              = collect();
        $this->cityList               = collect();

        // Reset category lists if using MemberCategoryTrait
        if (method_exists($this, 'resetCategoryFilters')) {
            $this->subcategoriesList      = collect();
            $this->specificCategoriesList = collect();
        }

        // Apply search after clear all filters
        $this->applySearch();
    }

    public function applySortOrder()
    {
        $this->parseSortOrder();
        $this->applySearch();
    }

    public function parseSortOrder()
    {
        if ($this->sort == 'created-desc') {
            $this->sortOrder = 'created_at';
            $this->order     = 'desc';
            $this->sortText  = 'Most Recent';
        } elseif ($this->sort == 'created-asc') {
            $this->sortOrder = 'created_at';
            $this->order     = 'asc';
            $this->sortText  = 'Most Recent';
        } elseif ($this->sort == 'relevent') {
            $this->sortOrder = 'created_at';
            $this->order     = 'desc';
            $this->sortText  = 'Most Relevant';
        } elseif ($this->sort == 'rating-desc') {
            $this->sortOrder = 'profile_rating';
            $this->order     = 'desc';
            $this->sortText  = 'Highest Rated';
        } elseif ($this->sort == 'rating-asc') {
            $this->sortOrder = 'profile_rating';
            $this->order     = 'asc';
            $this->sortText  = 'Lowest Rated';
        } elseif ($this->sort == 'featured-first') {
            // dump($this->sort);
            $this->sortOrder = 'featured_first';
            // dump($this->sortOrder);
            $this->order     = 'desc';
            $this->sortText  = 'Featured First';
        } else {
            $this->sortOrder = 'created_at';
            $this->order     = 'desc';
            $this->sortText  = 'Best Match';
        }
    }

    /**
     * Parse location string into lat/lng coordinates
     */
    private function parseLocationString()
    {
        if (! empty($this->location)) {
            $splitLocation = explode(',', $this->location);
            if (count($splitLocation) >= 2) {
                $this->lat = trim($splitLocation[0]);
                $this->lng = trim($splitLocation[1]);
            }
        }
    }

    public function applySearch()
    {
        // If we have lat/lng but no location string, set it
        if (! empty($this->lat) && ! empty($this->lng) && empty($this->location)) {
            $this->location = $this->lat . ',' . $this->lng;
        }

        $this->generateSearchForText();
        $this->resetPage();
    }

    /**
     * Listen for search-members event and update search parameters
     */
    #[On('search-members')]
    public function updateSearchParams($params)
    {
        $this->resetPage();
        $this->searchParams = $params;
    }

    public function getCountryList()
    {
        $this->countryList = $this->countryService->getAll()->get();
    }

    public function getRadiusList()
    {
        $defaultRadiusList = [
            ['id' => 10, 'name' => '10 miles'],
            ['id' => 25, 'name' => '25 miles'],
            ['id' => 50, 'name' => '50 miles'],
            ['id' => 100, 'name' => '100 miles'],
            ['id' => 250, 'name' => '250 miles'],
            ['id' => 'custom', 'name' => 'Custom Radius'],
        ];
        $settingsRadiusList = get_general_setting('member_search_radius_list', $defaultRadiusList);
        $this->radiusList   = collect($settingsRadiusList);
    }

    public function getProfileFeatures()
    {
        $features = [
            ['id' => 'reviews', 'name' => 'With Customer Reviews'],
            ['id' => 'emergency', 'name' => 'Emergency (24/7)'],
            ['id' => 'open_now', 'name' => 'Open Now'],
        ];
        $profileFeatures       = get_general_setting('profile_features', $features);
        $this->profileFeatures = collect($profileFeatures);
    }

    /**
     * Get rating data with real counts from UserService
     */
    public function getRatingData(): array
    {
        // Get real counts from UserService
        $ratingCounts = $this->userService->getUserCountsByRating();

        return [
            ['value' => 1, 'label' => '1 star', 'count' => $ratingCounts[1] ?? 0],
            ['value' => 2, 'label' => '2 stars', 'count' => $ratingCounts[2] ?? 0],
            ['value' => 3, 'label' => '3 stars', 'count' => $ratingCounts[3] ?? 0],
            ['value' => 4, 'label' => '4 stars', 'count' => $ratingCounts[4] ?? 0],
            ['value' => 5, 'label' => '5 stars', 'count' => $ratingCounts[5] ?? 0],
        ];
    }

    /**
     * Get all states for selected country
     *
     * @param  string|null  $countrySlug  Country slug to get states for
     * @return Collection Collection of states
     */
    private function fetchStatesForCountry($countrySlug): Collection
    {
        $allStates = collect();

        // Make sure countrySlug is a string
        if (is_array($countrySlug)) {
            // If somehow we got an array, take the first value
            $countrySlug = ! empty($countrySlug) ? (string) reset($countrySlug) : '';
        }

        // If country is provided
        if (! empty($countrySlug) && is_string($countrySlug)) {
            $country = $this->countryService->getCountryBySlug($countrySlug);

            if ($country) {
                // Get states for this country
                $states = $this->stateService->getAll(['country_id' => $country->id])->get();

                // Ensure states is a collection
                $allStates = $states instanceof Collection ? $states : collect($states);
            }
        }

        // Return states
        return $allStates->unique('id')->values();
    }

    /**
     * Get all cities for selected states
     *
     * @param  array|Collection  $stateSlugs  State slugs to get cities for
     * @return Collection Collection of cities
     */
    private function fetchCitiesForStates($stateSlugs): Collection
    {
        $allCities = collect();

        // Convert to collection if not already
        $stateSlugs = collect($stateSlugs);

        // If states are provided
        if ($stateSlugs->isNotEmpty()) {
            // Process each state slug
            foreach ($stateSlugs as $slug) {
                $state = $this->stateService->getStateBySlug($slug);

                if ($state) {
                    // Get cities for this state
                    $cities = $this->cityService->getAll(['state_id' => $state->id])->get();

                    // Ensure cities is a collection
                    $cities = $cities instanceof Collection ? $cities : collect($cities);

                    // Merge with main collection
                    $allCities = $allCities->merge($cities);
                }
            }
        }

        // Return unique cities
        return $allCities->unique('id')->values();
    }

    /**
     * Filter selected locations based on what's available in the full location list
     *
     * @param  array|Collection  $selectedSlugs  Currently selected location slugs
     * @param  Collection  $availableLocations  Available locations to filter against
     * @param  string  $type  Type of location ('state' or 'city')
     * @return array Filtered selection array
     */
    private function filterSelectedLocations($selectedSlugs, Collection $availableLocations, string $type): array
    {
        $locationIdsToKeep = $availableLocations->pluck('id')->toArray();

        return collect($selectedSlugs)
            ->filter(function ($slug) use ($locationIdsToKeep, $type) {
                $location = null;
                if ($type === 'state') {
                    $location = $this->stateService->getStateBySlug($slug);
                } elseif ($type === 'city') {
                    $location = $this->cityService->getCityBySlug($slug);
                }

                return $location && in_array($location->id, $locationIdsToKeep);
            })
            ->toArray();
    }

    public function getStatesList()
    {
        // Get all possible states for selected country
        $allStates = $this->fetchStatesForCountry($this->country);

        // Filter selected states if we have any previous states
        if ($this->stateList instanceof Collection && $this->stateList->isNotEmpty()) {
            // If country changed, check if states need to be filtered
            if (empty($this->country) || $allStates->isEmpty()) {
                // Country was removed or has no states, clear states and cities
                $this->state     = '';
                $this->city      = '';
                $this->stateList = collect();
                $this->cityList  = collect();
            } else {
                // Filter selected states to only include those with parent country still selected
                $stateIdsToKeep = $allStates->pluck('id')->toArray();

                // Handle both string and array state values
                $currentState   = is_array($this->state) ? $this->state : [$this->state];
                $filteredStates = collect($currentState)
                    ->filter(function ($stateSlug) use ($stateIdsToKeep) {
                        if (empty($stateSlug)) {
                            return false;
                        }
                        $state = $this->stateService->getStateBySlug($stateSlug);

                        return $state && in_array($state->id, $stateIdsToKeep);
                    })
                    ->toArray();

                // Convert back to string if it was originally a string
                $this->state = ! is_array($this->state) && count($filteredStates) > 0 ? $filteredStates[0] : '';

                // If all states were removed, clear cities as well
                if (empty($this->state)) {
                    $this->city     = '';
                    $this->cityList = collect();
                } else {
                    // Update cities based on remaining states
                    $this->getCitiesList();
                }
            }
        }

        // Update the state list
        $this->stateList = $allStates;
    }

    public function getCitiesList()
    {
        // Convert state to array for processing
        $stateArray = is_string($this->state) && ! empty($this->state) ? [$this->state] : [];

        // Get all possible cities for selected states
        $allCities = $this->fetchCitiesForStates($stateArray);

        // Filter selected cities if needed
        if (! empty($this->city) && $this->cityList instanceof Collection && $this->cityList->isNotEmpty()) {
            // Handle both string and array city values
            $currentCity    = is_array($this->city) ? $this->city : [$this->city];
            $filteredCities = $this->filterSelectedLocations($currentCity, $allCities, 'city');

            // Convert back to string if it was originally a string
            $this->city = ! is_array($this->city) && count($filteredCities) > 0 ? $filteredCities[0] : '';
        }

        // Update the city list
        $this->cityList = $allCities;
    }

    /**
     * Ensure properties are properly typed after Livewire hydration
     */
    public function hydrate()
    {
        // Ensure country is always a string, not an array
        if (is_array($this->country)) {
            $this->country = ! empty($this->country) ? (string) reset($this->country) : '';
        }

        // Ensure state is always a string, not an array
        if (is_array($this->state)) {
            $this->state = ! empty($this->state) ? (string) reset($this->state) : '';
        }

        // Ensure city is always a string, not an array
        if (is_array($this->city)) {
            $this->city = ! empty($this->city) ? (string) reset($this->city) : '';
        }

        // Re-parse location after hydration
        $this->parseLocationString();
    }


    public function toggle(string $section): void
    {
        $prop = 'open' . Str::studly($section);
        if (property_exists($this, $prop)) {
            $this->$prop = ! $this->$prop;
        }
    }

    /**
     * Prepare search arguments for querying members
     *
     * @param  bool  $includeListArgs  Whether to include list-specific arguments (limit, order_by, etc.)
     * @return array Formatted search arguments
     */
    protected function prepareSearchArgs(bool $includeListArgs = false): array
    {
        // Handle radius logic - use actual radius value when custom is selected
        $actualRadius = $this->radius === 'custom' ? $this->customRadius : $this->radius;

        // Base arguments used by both views
        $args = [
            'role'        => ESP_ROLE_MEMBER,
            'radius'      => $actualRadius,
            'search'      => $this->search,
            'country'     => $this->country,
            'state'       => $this->state,
            'city'        => $this->city,
            'rating'      => $this->rating,
            'location'    => $this->location,
            'lat'         => $this->lat,
            'lng'         => $this->lng,
        ];

        // Add list-specific arguments if requested
        if ($includeListArgs) {
            $args['limit']            = $this->perPage;
            $args['features']         = $this->features;
            $args['order_by']         = $this->sortOrder;
            $args['order']            = $this->order;
        }
        // Merge searchParams if they exist
        if (! empty($this->searchParams)) {
            $args = array_merge($args, $this->searchParams);
        }

        return $args;
    }

    public function generateSearchForText()
    {
        $searchText = '';
        $location   = '';

        if (! empty($this->search)) {
            $searchText .= $this->search;
        }

        // Add category text if using MemberCategoryTrait
        if (method_exists($this, 'generateCategorySearchText')) {
            $searchText .= $this->generateCategorySearchText();
        }

        if (! empty($this->rating)) {
            $searchText .= ' with ';
            $ratingsText = [];
            foreach ($this->rating as $rating) {
                $ratingsText[] = $rating;
            }
            $searchText .= implode(',', $ratingsText) . ' star rating';
        }

        if (! empty($this->city)) {
            $location .= ' ' . $this->city;
        } else {
            if (! empty($this->state)) {
                $stateData = $this->stateList->where('slug', $this->state)->first();
                if ($stateData) {
                    $location .= ' ' . $stateData->state_name;
                }
            }
            if (! empty($this->country)) {
                $countryCollection = collect($this->countryList);
                $countryData       = $countryCollection->where('slug', $this->country)->first();
                if ($countryData) {
                    $location .= ' ' . $countryData->country_name;
                }
            }
        }
        
        if (! empty($location)) {
            $searchText .= ' near' . $location;
        }

        $this->searchForTxt = $searchText;
    }

}
