<?php

namespace App\Livewire\Frontend\Member\Listing\SearchResult;

use App\Models\Location\Country;
use App\Services\Location\CityService;
use App\Services\Location\CountryService;
use App\Services\Location\StateService;
use App\Services\Review\ReviewService;
use App\Services\User\MemberCategoryService;
use App\Services\User\UserService;
use Livewire\Component;

class SearchResultMapView extends Component
{
    use MemberSearchTrait, MemberCategoryTrait;

    public $membersData = [];
    public $mapLoading = false;

    public function boot(
        UserService $userService,
        MemberCategoryService $memberCategoryService,
        CountryService $countryService,
        StateService $stateService,
        CityService $cityService,
        ReviewService $reviewService
    ) {
        $this->userService = $userService;
        $this->memberCategoryService = $memberCategoryService;
        $this->countryService = $countryService;
        $this->stateService = $stateService;
        $this->cityService = $cityService;
        $this->reviewService = $reviewService;
    }

    public function mount($page, $viewType, $breadcrumbs)
    {
        $this->page = $page;
        $this->viewType = $viewType ?? $page->data['member_list_template']['default_view'] ?? 'list';
        $this->breadcrumbs = $breadcrumbs;

        // Initialize empty collections
        $this->categoriesList = collect();
        $this->subcategoriesList = collect();
        $this->specificCategoriesList = collect();
        $this->stateList = collect();
        $this->cityList = collect();

        // Initialize lists
        $this->getParentCategories();
        $this->getChildCategories();
        $this->getSpecificCategories();
        $this->getCountryList();

        // Load states if country is selected
        if (! empty($this->country)) {
            $this->getStatesList();
            $countryModel = Country::where('slug', $this->country)->first();
            if ($countryModel) {
                $this->countryCode = $countryModel->country_code;
            }
        }

        // Load cities if states are selected
        if (! empty($this->state)) {
            $this->getCitiesList();
        }

        $this->getRadiusList();
        $this->getProfileFeatures();

        // Parse location string into lat/lng if provided
        $this->parseLocationString();

        $this->generateSearchForText();

        $this->parseSortOrder();
    }

    /**
     * Get both paginated and all members based on current filters
     *
     * @return array Contains 'paginated' and 'all' members
     */
    protected function getMembers(): array
    {
        // Include list args (true) to get sort parameters
        $args = $this->prepareSearchArgs(true);

        // Add category-related arguments
        $args['category'] = $this->category;
        $args['subcategory'] = $this->subcategory;
        $args['service'] = $this->service;
        $args['specificCategory'] = $this->specificCategory;

        $paginatedMembers = $this->userService->getMemberList($args);
        // Get a collection of members for the map - use only the visible items
        // to avoid redundant database queries
        $membersForMap = $paginatedMembers->items();

        return [
            'paginated' => $paginatedMembers,
            'items' => $membersForMap,
        ];
    }

    /**
     * Format member data for map display
     *
     * @param  array|Collection  $members  Array or Collection of member objects
     * @return array Formatted members for map display
     */
    protected function formatMembersForMap($members): array
    {
        return collect($members)->map(function (\App\Models\User\User $member) {
            // Format basic information needed for map markers
            $rating = $member->ratingCount();
            $categories = $member->parentCategories(1);
            $categoryName = ($categories->first()->category_name) ?? '';
            $profileLink = route('member.show', ['slug' => $member->slug]);

            // Format member data into a simple array structure
            $formattedMember = [
                'id' => (string) $member->id,
                'uuid' => (string) $member->uuid,
                'name' => (string) $member->memberName(),
                'title' => (string) $member->memberName(),
                'category' => (string) $categoryName,
                'city' => (string) ($member->primaryLocation->city->city_name ?? ''),
                'state' => (string) ($member->primaryLocation->state->state_name ?? ''),
                'country' => (string) ($member->primaryLocation->country->country_name ?? ''),
                'location' => (string) $member->address('area'),
                'lat' => (float) ($member->primaryLocation->latitude ?? 0),
                'lng' => (float) ($member->primaryLocation->longitude ?? 0),
                'rating' => (float) ($rating['rating'] ?? 0),
                'total_rating_count' => (int) ($rating['count'] ?? 0),
                'profile_image' => (string) $member->logoImage(),
                'link' => (string) $profileLink,
                'phone' => (string) ($member->phone ?? ''),
            ];

            // Generate HTML for the info window
            $formattedMember['infoWindowHtml'] = $this->generateMemberCardHtml($formattedMember);

            return $formattedMember;
        })->toArray();
    }

    /**
     * Set map loading state
     */
    public function setMapLoading($loading = true)
    {
        $this->mapLoading = $loading;
    }

    /**
     * Hook into updating lifecycle to set loading state
     */
    public function updating($name)
    {
        // Check if the page property is being updated (pagination)
        if ($name === 'paginators.page' || $name === 'page') {
            $this->mapLoading = true;
        }
    }

    /**
     * Hook into updated lifecycle to handle after pagination
     */
    public function updated($name)
    {
        // Check if the page property was updated
        if ($name === 'paginators.page' || $name === 'page') {
            // Loading will be reset in render method after markers are dispatched
        }
    }

    /**
     * Generate HTML for member card to be used in map info window
     *
     * @param  array  $member  The formatted member data
     * @return string The HTML for the member card
     */
    protected function generateMemberCardHtml(array $member): string
    {
        // Use the Blade component instead of generating HTML directly
        return view('components.frontend.member.listing.partials.map-info-window', [
            'member' => $member,
        ])->render();
    }

    public function render()
    {
        $members = $this->getMembers();

        // Dispatch member data for the map
        $membersForMap = $this->formatMembersForMap($members['items']);
        $this->membersData = $membersForMap;
        $this->dispatch('members-updated', members: $membersForMap);

        // Reset loading state after successful render
        if ($this->mapLoading) {
            $this->dispatch('map-loading-complete');
            $this->mapLoading = false;
        }

        return view('livewire.frontend.member.listing.search-result.search-result-map-view', [
            'members' => $members['paginated'],
        ]);
    }
}
