<?php

namespace App\Livewire\Frontend\Member\Modal;

use App\Events\ContactUs;
use App\Models\User\User;
use Illuminate\Support\Facades\Auth;
use LivewireUI\Modal\ModalComponent;
use App\Services\User\UserService;
use SnipifyDev\LaravelCaptcha\Rules\RecaptchaValidationRule;

class MemberContactModal extends ModalComponent
{
    public $member;
    public $name;
    public $email;
    public $phone;
    public $message;
    public $canContact = true;

    /** @var string - reCAPTCHA token */
    public $captchaToken = '';

    public function mount($member, UserService $userService)
    {
        // Find the member
        if(is_numeric($member)) {
            $this->member = $userService->find($member);
        } else {
            $this->member = $userService->findBySlug($member);
        }
        
        // Check if user is logged in
        if (Auth::check()) {
            $user = Auth::user();
            $this->name = $user->name;
            $this->email = $user->email;
            $this->phone = $user->phone;
            // Prevent contacting self
            if(is_numeric($member)) {
                if ($user->id === $member) {
                    $this->canContact = false;
                }
            } else {
                if ($user->id === $this->member->id) {
                    $this->canContact = false;
                }
            }
        }
    }

    public function sendMessage()
    {
        if (!$this->canContact) {
            session()->flash('error', 'You cannot contact yourself.');
            return;
        }

        // Validate all fields including captcha using Laravel validation
        $senderData = $this->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'phone' => 'required|string|max:255',
            'message' => 'required|string|max:255',
            'captchaToken' => ['required', new RecaptchaValidationRule('contact_modal', null, 'v3')],
        ]);

        // Add recipient and sender information to the data
        $data = array_merge($senderData, [
            'recipient_id' => is_numeric($this->member) ? $this->member : $this->member->id,
            'recipient_email' => $this->member->email,
            'sender_id' => Auth::check() ? Auth::id() : null,
        ]);

        event(new ContactUs($data));

        session()->flash('success', 'Message sent successfully!');
        $this->reset(['message']);
    }


    /**
     * Supported: 'sm', 'md', 'lg', 'xl', '2xl', '3xl', '4xl', '5xl', '6xl', '7xl'
     */
    public static function modalMaxWidth(): string
    {
        return 'xl';
    }

    public static function closeModalOnEscape(): bool
    {
        return true;
    }

    public static function closeModalOnClickAway(): bool
    {
        return true;
    }

    public static function closeModalOnEscapeIsForceful(): bool
    {
        return false;
    }

    public function render()
    {
        return view('livewire.frontend.member.modal.member-contact-modal');
    }
}
