<?php

namespace App\Livewire\Frontend\Member\Modal;

use LivewireUI\Modal\ModalComponent;
use App\Services\User\UserService;
use App\Services\Review\ReviewService;
use App\Enums\MemberRatingType;
use App\Models\Review;
use App\Models\ReviewRating;
use Livewire\WithFileUploads;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class ReviewWriteModal extends ModalComponent
{
    use WithFileUploads;

    public $member;
    public $currentStep = 1;

    // Overall Experience Fields
    public $overallRating = 0;
    public $reviewTitle = '';
    public $reviewDescription = '';

    // How Things Went Fields (Ratings)
    public $serviceQualityRating = 0;
    public $responseTimeRating = 0;
    public $professionalismRating = 0;
    public $valueForMoneyRating = 0;
    public $communicationRating = 0;

    // Photo Upload Fields
    public $photos = [];
    public $uploadedPhotos = [];

    // Form State
    public $showSuccessModal = false;
    protected $reviewService;

    protected $rules = [
        'overallRating' => 'required|integer|min:1|max:5',
        'reviewTitle' => 'nullable|string|max:255',
        'reviewDescription' => 'required|string|min:50',
        'serviceQualityRating' => 'required|integer|min:1|max:5',
        'responseTimeRating' => 'required|integer|min:1|max:5',
        'professionalismRating' => 'required|integer|min:1|max:5',
        'valueForMoneyRating' => 'required|integer|min:1|max:5',
        'communicationRating' => 'required|integer|min:1|max:5',
        'photos.*' => 'nullable|image|max:5120', // 5MB max
    ];

    protected $messages = [
        'overallRating.required' => 'Please select an overall rating.',
        'overallRating.min' => 'Rating must be at least 1 star.',
        'reviewDescription.required' => 'Please provide a review description.',
        'reviewDescription.min' => 'Review description must be at least 50 characters.',
        'serviceQualityRating.required' => 'Please rate the service quality.',
        'responseTimeRating.required' => 'Please rate the response time.',
        'professionalismRating.required' => 'Please rate the professionalism.',
        'valueForMoneyRating.required' => 'Please rate the value for money.',
        'communicationRating.required' => 'Please rate the communication.',
        'photos.*.image' => 'Each file must be an image.',
        'photos.*.max' => 'Each image must not exceed 5MB.',
    ];

    public function boot(ReviewService $reviewService)
    {
        $this->reviewService = $reviewService;
    }

    public function mount($member, UserService $userService)
    {

        $this->member = $userService->findBySlug($member);
    }

    public function isAuthenticated()
    {
        return Auth::check();
    }

    public function canWriteReview()
    {
        if (!Auth::check()) {
            return false;
        }

        // Prevent users from writing reviews about themselves
        return Auth::id() !== $this->member->id;
    }

    public function nextStep()
    {
        if ($this->currentStep === 1) {
            $this->validateStep1();
        } elseif ($this->currentStep === 2) {
            $this->validateStep2();
        } elseif ($this->currentStep === 3) {
            // No validation needed for photo upload step
        }

        if ($this->currentStep < 4) {
            $this->currentStep++;
        }
    }

    public function previousStep()
    {
        if ($this->currentStep > 1) {
            $this->currentStep--;
        }
    }

    public function goToStep($step)
    {
        if ($step >= 1 && $step <= 4) {
            // Validate previous steps before jumping ahead
            if ($step > $this->currentStep) {
                for ($i = $this->currentStep; $i < $step; $i++) {
                    if ($i === 1) {
                        $this->validateStep1();
                    } elseif ($i === 2) {
                        $this->validateStep2();
                    }
                }
            }
            $this->currentStep = $step;
        }
    }

    protected function validateStep1()
    {
        $this->validate([
            'overallRating' => 'required|integer|min:1|max:5',
            'reviewTitle' => 'nullable|string|max:255',
            'reviewDescription' => 'required|string|min:50',
        ]);
    }

    protected function validateStep2()
    {
        $this->validate([
            'serviceQualityRating' => 'required|integer|min:1|max:5',
            'responseTimeRating' => 'required|integer|min:1|max:5',
            'professionalismRating' => 'required|integer|min:1|max:5',
            'valueForMoneyRating' => 'required|integer|min:1|max:5',
            'communicationRating' => 'required|integer|min:1|max:5',
        ]);
    }

    public function updatedPhotos()
    {
        $this->validate([
            'photos.*' => 'image|max:5120',
        ]);

        // Check total photo count
        if (count($this->uploadedPhotos) + count($this->photos) > 5) {
            $this->addError('photos', 'You can upload a maximum of 5 photos.');
            $this->photos = [];
            return;
        }

        foreach ($this->photos as $photo) {
            $this->uploadedPhotos[] = [
                'file' => $photo,
                'temporaryUrl' => $photo->temporaryUrl(),
            ];
        }

        $this->photos = [];
    }

    public function removePhoto($index)
    {
        unset($this->uploadedPhotos[$index]);
        $this->uploadedPhotos = array_values($this->uploadedPhotos);
    }

    public function getRatingText($rating)
    {
        return match ($rating) {
            1 => 'Bad',
            2 => 'Good',
            3 => 'Pretty good',
            4 => 'Excellent',
            5 => 'Loved it!',
            default => ''
        };
    }

    public function submitReview()
    {
        // Prevent self-review on backend as well
        if (!$this->canWriteReview()) {
            $this->addError('submit', 'You cannot write a review for your own profile.');
            return;
        }

        $this->validate();

        try {
            DB::beginTransaction();

            // Create the main review
            $review = Review::create([
                'review_rating' => $this->overallRating,
                'review_title' => $this->reviewTitle ?: null,
                'review_content' => $this->reviewDescription,
                'review_user_id' => Auth::id(),
                'reviewable_type' => 'App\Models\User\User',
                'reviewable_id' => $this->member->id,
                'status' => true,
            ]);

            // Create rating records
            $ratings = [
                ['rating_type' => MemberRatingType::ServiceQuality->value, 'rating' => $this->serviceQualityRating],
                ['rating_type' => MemberRatingType::ResponseTime->value, 'rating' => $this->responseTimeRating],
                ['rating_type' => MemberRatingType::Professionalism->value, 'rating' => $this->professionalismRating],
                ['rating_type' => MemberRatingType::ValueForMoney->value, 'rating' => $this->valueForMoneyRating],
                ['rating_type' => MemberRatingType::Communication->value, 'rating' => $this->communicationRating],
            ];

            foreach ($ratings as $rating) {
                $review->ratings()->create($rating);
            }

            // Handle photo uploads
            foreach ($this->uploadedPhotos as $photoData) {
                $review->addMedia($photoData['file']->getRealPath())
                    ->toMediaCollection('review_attachments');
            }

            DB::commit();
            $this->dispatch('reviewSubmitted');
            // Show success modal briefly
            $this->showSuccessModal = true;
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Review submission failed: ' . $e->getMessage());
            $this->addError('submit', 'Failed to submit review. Please try again.');
        }
    }

    /**
     * Supported: 'sm', 'md', 'lg', 'xl', '2xl', '3xl', '4xl', '5xl', '6xl', '7xl'
     */
    public static function modalMaxWidth(): string
    {
        return '4xl';
    }

    public static function closeModalOnEscape(): bool
    {
        return true;
    }

    public static function closeModalOnClickAway(): bool
    {
        return true;
    }

    public static function closeModalOnEscapeIsForceful(): bool
    {
        return false;
    }

    public function render()
    {
        return view('livewire.frontend.member.modal.review-write-modal');
    }
}
