<?php

namespace App\Livewire\Frontend\Widgets;

use Livewire\Component;
use App\Services\Job\JobService;
use App\Services\Job\JobRoleService;
use App\Models\Job\JobRole;
use App\Models\Job\JobType;
use App\Models\Job\JobCareerLevel;

class JobList extends Component
{
    protected $component;
    protected $jobService;
    protected $jobRoleService;
    public $limit = 3;
    public $heading;
    public $description;
    public $jobs;

    

    public function boot(JobService $jobService, JobRoleService $jobRoleService)
    {
        $this->jobService = $jobService;
        $this->jobRoleService = $jobRoleService;
    }

    public function mount($pageData = [])
    {
        $component = $pageData['job_list'];
        $this->component = $component;
        $this->heading = !empty($component['heading']) ? $component['heading'] : '<p>Search or list <em>jobs in Vancouver</em></p>';
        $this->description = !empty($component['description']) ? $component['description'] : 'Jobs';
        $this->limit = !empty($component['limit']) ? $component['limit'] : $this->limit;
    }


    public function getTopJobs(){
        
        $componentData = $this->component ?? [];
        // 1) Selected jobs: fetch exactly by provided IDs
        if (!empty($componentData['selected_jobs'])) {
            $selectedIds = $componentData['selectedJobIds'] ?? $componentData['jobs'] ?? [];
            $selectedIds = array_values(array_filter(array_map('intval', (array)$selectedIds)));

            if (!empty($selectedIds)) {
                $this->jobs = $this->jobService
                    ->getAll()
                    ->whereIn('jobs.id', $selectedIds)
                    ->get();
                return;
            }
        }

        // 2) Latest jobs with filters from $componentData
        if (!empty($componentData['latest_jobs'])) {
            $args = [
                'limit' => $this->limit,
                'order_by' => 'created_at',
                'order' => 'desc',
            ];
            $hasFilters = false; // Track presence of selected filters listed by user

            // Location: accept either IDs or direct keys
            if (!empty($componentData['city_id'])) {
                $args['city'] = (int) $componentData['city_id'];
                $hasFilters = true;
            } elseif (!empty($componentData['state_id'])) {
                $args['state'] = (int) $componentData['state_id'];
                $hasFilters = true;
            }
            if (!empty($componentData['country_id'])) {
                $args['country'] = (int) $componentData['country_id'];
                $hasFilters = true;
            }
            // Direct keys if provided
            if (!empty($componentData['city'])) {
                $args['city'] = $componentData['city'];
                $hasFilters = true;
            }
            if (!empty($componentData['state'])) {
                $args['state'] = $componentData['state'];
                $hasFilters = true;
            }
            if (!empty($componentData['country'])) {
                $args['country'] = $componentData['country'];
                $hasFilters = true;
            }
            if (!empty($componentData['contry'])) { // tolerate misspelling
                $args['country'] = $componentData['contry'];
                $hasFilters = true;
            }


            // Role / Type / Career level: repository expects slugs
            if (!empty($componentData['job_role'])) {
                $roleIds = (array) $componentData['job_role'];
                $args['job_role'] = $roleIds;
            }

            if (!empty($componentData['job_type_id'])) {
                $typeIds = (array) $componentData['job_type_id'];
                $args['job_type'] = JobType::query()
                    ->whereIn('id', $typeIds)
                    ->pluck('slug')
                    ->filter()
                    ->values()
                    ->all();
                if (!empty($args['job_type'])) {
                    $hasFilters = true;
                }
            }
            if (!empty($componentData['job_type'])) {
                $args['job_type'] = (array) $componentData['job_type'];
                $hasFilters = true;
            }

            if (!empty($componentData['job_career_level_id'])) {
                $careerIds = (array) $componentData['job_career_level_id'];
                $args['career_level'] = JobCareerLevel::query()
                    ->whereIn('id', $careerIds)
                    ->pluck('slug')
                    ->filter()
                    ->values()
                    ->all();
                if (!empty($args['career_level'])) {
                    $hasFilters = true;
                }
            }
            if (!empty($componentData['experince'])) { // map to career_level
                $args['career_level'] = (array) $componentData['experince'];
                $hasFilters = true;
            }

            // Work mode / Shift
            if (!empty($componentData['work_mode'])) {
                $args['work_mode'] = (array) $componentData['work_mode'];
                $hasFilters = true;
            }
            if (!empty($componentData['job_shift'])) {
                $args['shift'] = (array) $componentData['job_shift'];
                $hasFilters = true;
            }
            if (!empty($componentData['shift'])) {
                $args['shift'] = (array) $componentData['shift'];
                $hasFilters = true;
            }

            // If latest_jobs with no selected filters, show only open jobs (apply_by_date >= today)
            if (! $hasFilters) {
                $args['open_jobs_only'] = true;
            }

            // Salary
            if (isset($componentData['min_salary'])) {
                $args['min_salary'] = $componentData['min_salary'];
            }
            if (isset($componentData['max_salary'])) {
                $args['max_salary'] = $componentData['max_salary'];
            }

            // Date posted mapping to days
            if (!empty($componentData['date_posted'])) {
                $map = [
                    '24_hours' => 1,
                    '7_days' => 7,
                    '30_days' => 30,
                    'any' => null,
                ];
                $days = $map[$componentData['date_posted']] ?? null;
                if (!empty($days)) {
                    $args['date_posted'] = $days;
                }
            }
            $this->jobs = $this->jobService->getAll($args)->get();
            return;
        }

        // 3) Default: latest jobs limited
        $args = [
            'limit' => $this->limit,
            'order_by' => 'deadline',
            'order' => 'desc',
        ];


        return $this->jobs = $this->jobService->getAll($args)->get();
    }

    public function render()
    {
        $this->getTopJobs();
        return view('livewire.frontend.widgets.job-list');
    }
}
