<?php

namespace App\Livewire\Frontend\Widgets;

use App\DTOs\Lead\Lead\CreateLeadDTO;
use App\Enums\PreferredContactWindow;
use App\Enums\PrimaryUse;
use App\Enums\PropertyRelationship;
use App\Enums\WaterQualityGoals;
use App\Services\Lead\LeadAddressService;
use App\Services\Lead\LeadService;
use Illuminate\Support\Facades\DB;
use Livewire\Component;
use Livewire\WithFileUploads;

class LeadForm extends Component
{
    use WithFileUploads;

    protected $pageData;

    public $heading;

    public $content;

    public $lead_form_id = null;

    // Location data from URL
    public $address;

    public $lat;

    public $lng;

    public $city;

    public $state;

    public $state_code;

    public $country;

    public $country_code;

    // Form fields
    public $title;

    public $description;

    public $message;

    public $firstName;

    public $lastName;

    public $emailAddress;

    public $phoneNumber;

    public $propertyRelationship = null;

    public $propertyRelationshipList = [];

    public $preferredContactWindow = null;

    public $preferredContactWindowList = [];

    public $authorizeToContactMe = false;

    public $primaryUse = [];

    public $primaryUseList = [];

    // Nested fields for each primary use type
    public $domestic_no_of_people = null;
    public $domestin_no_of_bathrooms = null;
    public $domestic_swimming_pool_garden = false;
    
    public $agriculture_irrigation_purpose = null;
    public $agriculture_area_size = null;
    
    public $commercial_business_detail = null;
    
    public $other_intented_use_detail = null;

    public $waterQualityGoals = [];

    public $waterQualityGoalsList = [];

    public $isExistingWaterWell;

    public $informationAboutExistingWell = null;

    public $isNewConstructionProject;

    public $desiredProjectTimeline = null;

    public $wellSiteLocation = null;

    public $desiredProjectTimelineList = [];

    public $completeEstimate = 0;

    public $notes = null;

    public $propertyLocation = 0;

    // File uploads
    public $attachments = [];

    public $newAttachments = [];

    // Wizard step state (0-based: 0..3)
    public int $currentStep = 0;

    protected LeadService $leadService;

    protected LeadAddressService $leadAddressService;

    public function boot(
        LeadService $leadService,
        LeadAddressService $leadAddressService
    ): void {
        $this->leadService = $leadService;
        $this->leadAddressService = $leadAddressService;
    }

    public function mount($pageData = [], $lead_form_id = null, $locationData = []): void
    {

        $this->pageData = $pageData;
        $this->heading = ! empty($this->pageData['heading']) ? $this->pageData['heading'] : 'Request your well drilling estimate';

        // Populate location data from URL parameters
        $this->address = $locationData['address'] ?? null;
        $this->lat = $locationData['lat'] ?? null;
        $this->lng = $locationData['lng'] ?? null;
        $this->city = $locationData['city'] ?? null;
        $this->state = $locationData['state'] ?? null;
        $this->state_code = $locationData['state_code'] ?? null;
        $this->country = $locationData['country'] ?? null;
        $this->country_code = $locationData['country_code'] ?? null;

        // Update content to show the address if provided
        if (! empty($this->address)) {
            $this->content = 'Creating estimate for: '.$this->address;
        } else {
            $this->content = ! empty($this->pageData['content']) ? $this->pageData['content'] : 'Creating estimate for: 123 Main Street, Fraser Valley, BC,hh V2S 1A1 ';
        }

        $this->primaryUseList = PrimaryUse::cases();
        $this->waterQualityGoalsList = WaterQualityGoals::cases();
        $this->propertyRelationshipList = PropertyRelationship::cases();
        $this->preferredContactWindowList = PreferredContactWindow::cases();
        // // Accept optional lead_form_id from caller
        $this->lead_form_id = ! empty($lead_form_id) ? $lead_form_id : null;
        $this->completeEstimate = 0;
        $this->getDesiredProjectTimeline();
    }

    protected function rules(): array
    {
        // Global rules (final submit). Keep minimal until more fields are wired.
        return [
            'lead_form_id' => 'nullable|exists:lead_forms,id',
            'firstName' => 'required|string|min:2|max:100',
            'lastName' => 'required|string|min:2|max:100',
            'emailAddress' => 'required|email:rfc,dns|max:255',
            'phoneNumber' => 'required|string|min:7|max:30',
            'authorizeToContactMe' => 'required|boolean',
            'attachments.*' => 'nullable|file|max:10240|mimes:jpeg,jpg,png,gif,webp,svg,pdf',
            'attachments' => 'nullable|array|max:5',
        ];
    }

    protected function stepRules(int $step): array
    {
        // Per-step validation. Currently step 0 (Contact Info) is required.
        return match ($step) {
            0 => [
                'firstName' => 'required|string|min:2|max:100',
                'lastName' => 'required|string|min:2|max:100',
                'emailAddress' => 'required|email:rfc,dns|max:255',
                'phoneNumber' => 'required|string|min:7|max:30',
                'propertyRelationship' => 'required|string',
                'preferredContactWindow' => 'required|string',
                'authorizeToContactMe' => 'required',
            ],
            1 => [
                'primaryUse' => 'required|array',
                'domestic_no_of_people' => $this->shouldValidateDomestic() ? 'required|string' : 'nullable',
                'domestin_no_of_bathrooms' => $this->shouldValidateDomestic() ? 'required|string' : 'nullable',
                'agriculture_irrigation_purpose' => $this->shouldValidateAgriculture() ? 'nullable|string' : 'nullable',
                'agriculture_area_size' => $this->shouldValidateAgriculture() ? 'nullable|string' : 'nullable',
                'commercial_business_detail' => $this->shouldValidateCommercial() ? 'nullable|string' : 'nullable',
                'other_intented_use_detail' => $this->shouldValidateOther() ? 'nullable|string' : 'nullable',
                'waterQualityGoals' => 'required|array',
            ],
            2 => [
                'isExistingWaterWell' => 'required|in:0,1',
                'informationAboutExistingWell' => 'nullable|string|max:1000',
                'isNewConstructionProject' => 'required|in:0,1',
                'desiredProjectTimeline' => 'required',
                'wellSiteLocation' => 'required|string|max:1000',
            ],
            3 => [
                'completeEstimate' => 'required|in:0,1',
                'notes' => 'nullable|string|max:1000',
                'propertyLocation' => 'required|in:0,1',
                'attachments.*' => 'nullable|file|max:10240|mimes:jpeg,jpg,png,gif,webp,svg,pdf',
                'attachments' => 'nullable|array|max:5',
            ],
            default => []
        };
    }

    public function nextStep(): void
    {
        // Validate only the current step before moving forward
        $rules = $this->stepRules($this->currentStep);
        if (! empty($rules)) {
            $this->validate($rules);
        }

        if ($this->currentStep < 3) {
            $this->currentStep++;
            $this->dispatch('leadform-go-next');
        }
    }

    public function prevStep(): void
    {
        if ($this->currentStep > 0) {
            $this->currentStep--;
            $this->dispatch('leadform-go-prev');
        }
    }

    public function getDesiredProjectTimeline()
    {
        $defaultDesiredProjectTimeline = [
            [
                'id' => 1,
                'value' => '1 week',
                'name' => '1 Week',
            ],
            [
                'id' => 2,
                'value' => '2 weeks',
                'name' => '2 Weeks',
            ],
            [
                'id' => 3,
                'value' => '3 weeks',
                'name' => '3 Weeks',
            ],
            [
                'id' => 4,
                'value' => '4 weeks',
                'name' => '4 Weeks',
            ],
        ];
        // $settingsDesiredProjectTimeline = get_general_setting('desired_project_timeline', $defaultDesiredProjectTimeline);
        $this->desiredProjectTimelineList = collect($defaultDesiredProjectTimeline);
    }

    public function updatedNewAttachments(): void
    {
        // Validate newly uploaded files
        $this->validate([
            'newAttachments.*' => 'nullable|file|max:10240|mimes:jpeg,jpg,png,gif,webp,svg,pdf',
        ]);

        // Check if adding new files would exceed the limit
        $totalCount = count($this->attachments) + count($this->newAttachments);
        if ($totalCount > 5) {
            $this->addError('newAttachments', 'You can only upload a maximum of 5 files total.');
            $this->newAttachments = [];

            return;
        }

        // Append new files to existing attachments
        foreach ($this->newAttachments as $file) {
            $this->attachments[] = $file;
        }

        // Clear the temporary array
        $this->newAttachments = [];
    }

    public function removeAttachment($index): void
    {
        if (isset($this->attachments[$index])) {
            unset($this->attachments[$index]);
            $this->attachments = array_values($this->attachments);
        }
    }

    protected function getFieldsForPrimaryUse(string $useType): array
    {
        return match($useType) {
            'residential_domestic' => [
                'domestic_no_of_people' => $this->domestic_no_of_people,
                'domestin_no_of_bathrooms' => $this->domestin_no_of_bathrooms,
                'domestic_swimming_pool_garden' => (bool) $this->domestic_swimming_pool_garden,
            ],
            'irrigation_agricultural' => [
                'agriculture_irrigation_purpose' => $this->agriculture_irrigation_purpose,
                'agriculture_area_size' => $this->agriculture_area_size,
            ],
            'commercial_business' => [
                'commercial_business_detail' => $this->commercial_business_detail,
            ],
            'other' => [
                'other_intented_use_detail' => $this->other_intented_use_detail,
            ],
            default => [],
        };
    }

    protected function shouldValidateDomestic(): bool
    {
        return in_array('residential_domestic', $this->primaryUse);
    }

    protected function shouldValidateAgriculture(): bool
    {
        return in_array('irrigation_agricultural', $this->primaryUse);
    }

    protected function shouldValidateCommercial(): bool
    {
        return in_array('commercial_business', $this->primaryUse);
    }

    protected function shouldValidateOther(): bool
    {
        return in_array('other', $this->primaryUse);
    }

    public function submit()
    {
        // Validate all required inputs for final submission
        $this->validate();

        // Build nested primary use array
        $primaryUseData = [];
        foreach ($this->primaryUse as $useType) {
            $primaryUseData[$useType] = $this->getFieldsForPrimaryUse($useType);
        }

        $meta = [
            'property_relationship' => $this->propertyRelationship,
            'preferred_contact_window' => $this->preferredContactWindow,
            'authorize_to_contact_me' => (bool) $this->authorizeToContactMe,
            'primary_use' => $primaryUseData,
            'water_quality_goals' => $this->waterQualityGoals,
            'is_existing_water_well' => $this->isExistingWaterWell,
            'information_about_existing_well' => $this->informationAboutExistingWell,
            'is_new_construction_project' => $this->isNewConstructionProject,
            'desired_project_timeline' => $this->desiredProjectTimeline,
            'well_site_location' => $this->wellSiteLocation,
            'complete_estimate' => (int) $this->completeEstimate,
            'notes' => $this->notes,
            'confirm_property_location' => (int) $this->propertyLocation,
            'title' => $this->title,
            'description' => $this->description,
        ];

        $dto = new CreateLeadDTO([
            'lead_form_id' => $this->lead_form_id ? (int) $this->lead_form_id : null,
            'lead_source' => 'frontend:lead_form_widget',
            'message' => $this->message,
            'first_name' => $this->firstName,
            'last_name' => $this->lastName,
            'email' => $this->emailAddress,
            'phone' => $this->phoneNumber,
            'lead_data' => [],
            'meta' => $meta,
        ]);

        DB::beginTransaction();
        try {
            $lead = $this->leadService->create($dto);

            // Create lead address if location data is provided
            $locationData = [
                'address' => $this->address,
                'lat' => $this->lat,
                'lng' => $this->lng,
                'city' => $this->city,
                'state' => $this->state,
                'state_code' => $this->state_code,
                'country' => $this->country,
                'country_code' => $this->country_code,
            ];

            $this->leadAddressService->createFromLocationData($lead->id, $locationData);

            // Attach uploaded files to lead
            if (! empty($this->attachments)) {
                foreach ($this->attachments as $attachment) {
                    $lead->addMedia($attachment->getRealPath())
                        ->usingFileName($attachment->getClientOriginalName())
                        ->toMediaCollection('attachments');
                }
            }

            DB::commit();

            session()->flash('success', __('Your estimate request has been submitted.'));

            // Reset form fields after successful submission
            $this->reset();

            return redirect()->route('lead.lead-summary', $lead->uuid);
        } catch (\Throwable $e) {
            DB::rollBack();
            report($e);
            session()->flash('error', __('Something went wrong. Please try again.'));
        }
    }

    public function render()
    {
        // Ensure the wizard UI re-initializes to the current step after any update
        $this->dispatch('leadform-refresh');

        return view('livewire.frontend.widgets.lead-form');
    }
}
