<?php

namespace App\Livewire\Frontend\Widgets;

use App\Enums\MembershipType;
use App\Models\User\MemberCategory;
use App\Services\User\UserService;
use Livewire\Component;

class TopMembers extends Component
{
    protected $userService;
    public $heading;
    public $sub_heading;
    public $description;
    public $limit = 10;
    public $isFeatured = false;
    public $members;
    public $useSelectedMembers = false;
    public $useLatestMembers = false;
    public $selectedMemberIds = [];
    public $selectedCategorySlug = null;
    public $categories = [];
    public $componentData = [];
    public $memberSelectionType = null;
    public $selectedCategories = [];

    /**
     * Boot the component
     */
    public function boot(UserService $userService)
    {
        $this->userService = $userService;
    }

    /**
     * Mount the component
     */
    public function mount($pageData = [])
    {
		$component = $pageData['top_members'];
		
		$this->componentData = $component;
		$this->heading = !empty($component['heading']) ? $component['heading'] : '<p>ESP <em>top rated companies</em> near you in <em>Vancouver</em></p>';
		$this->description = !empty($component['description']) ? $component['description'] : 'Want the top spot, get featured now.';
		$this->limit = !empty($component['limit']) ? (int) $component['limit'] : $this->limit;
		$this->isFeatured = (bool)($component['is_featured'] ?? false);
		$this->selectedMemberIds = is_array($component['members'] ?? null) ? array_map('intval', $component['members']) : [];
		
		// Determine member selection type (supports new radio field)
		$this->memberSelectionType = $component['member_selection_type']
			?? (($component['use_selected_members'] ?? false) ? 'use_selected_members' : 'use_latest_members');
		$this->useSelectedMembers = $this->memberSelectionType === 'use_selected_members';
		$this->useLatestMembers = $this->memberSelectionType === 'use_latest_members';
		
		// Get selected categories from component data
		$this->selectedCategories = !empty($component['categories']) && is_array($component['categories']) 
			? array_map('intval', $component['categories']) 
			: [];
		
		// Load categories for tabs (only parent categories that were selected)
		if ($this->memberSelectionType === 'use_latest_members' && !empty($this->selectedCategories)) {
			$this->categories = MemberCategory::whereNull('parent_id')
				->whereIn('id', $this->selectedCategories)
				->with('children')
				->orderBy('category_name', 'asc')
				->get();
		} else {
			$this->categories = collect();
		}
		
		// Load initial members
		$this->getTopMembers($component);
    }

    /**
     * Get top members
     */
	public function getTopMembers(array $componentData = [])
	{
		
		// If using selected members, fetch exactly those IDs (ignore other filters)
		if ($this->useSelectedMembers && ! empty($this->selectedMemberIds)) {
			$this->members = $this->userService
				->getAll(['role' => ESP_ROLE_MEMBER])
				->active()
				->whereIn('users.id', $this->selectedMemberIds)
				->limit($this->limit)
				->get();
			return;
		}

		// Build args based on latest members and provided filters
		$args = [
			'role' => ESP_ROLE_MEMBER,
		];

		if ($this->useLatestMembers) {
			// Categories (IDs)
			if (! empty($componentData['categories']) && is_array($componentData['categories'])) {
				$args['categories'] = array_map('intval', $componentData['categories']);
			}
			if (! empty($componentData['subcategories']) && is_array($componentData['subcategories'])) {
				$args['subcategories'] = array_map('intval', $componentData['subcategories']); 
			}
			if (! empty($componentData['services']) && is_array($componentData['services'])) {
				$args['services'] = array_map('intval', $componentData['services']);
			}

			// Location (IDs)
			if (! empty($componentData['city_id'])) {
				$args['city'] = (int) $componentData['city_id'];
			} elseif (! empty($componentData['state_id'])) {
				$args['state'] = (int) $componentData['state_id'];
			}
			if (! empty($componentData['country_id'])) {
				$args['country'] = (int) $componentData['country_id'];
			}

			// Rating
			if (!empty($componentData['rating'])) {
				$args['rating'] = (int) $componentData['rating'];
			}

			// Features
			$features = [];
			if (! empty($componentData['with_reviews'])) {
				$features[] = 'reviews';
			}
			if (! empty($componentData['emergency_only'])) {
				$features[] = 'emergency';
			}
			if (! empty($features)) {
				$args['features'] = $features;
			}

			// Featured filter
			if ($this->isFeatured) {
				$args['membershipType'] = MembershipType::Featured->value;
			}
		}

		// Apply selected category filter if set (includes parent + all children)
		if ($this->selectedCategorySlug) {
			$args['categories'] = $this->getCategoryIdsForFilter($this->selectedCategorySlug);
		}

		$this->members = $this->userService
			->getAll($args)
			->active()
			->orderBy('created_at', 'desc')
			->limit($this->limit)
			->get();
	}

	/**
	 * Load members for a specific category by slug
	 * This includes both the parent category and all its children
	 */
	public function loadCategoryMembers($categorySlug = null)
	{
		// If categorySlug is null (All tab), clear the filter
		if ($categorySlug === null) {
			$this->selectedCategorySlug = null;
		} else {
			$this->selectedCategorySlug = $categorySlug;
		}
		
		$this->getTopMembers($this->componentData);
	}
	
	/**
	 * Get all category IDs including children for filtering
	 * This returns both parent and all child category IDs so members
	 * associated with any of them will be included in results
	 * 
	 * @param string $categorySlug
	 * @return array
	 */
	protected function getCategoryIdsForFilter($categorySlug)
	{
		// Find the parent category by slug
		$parentCategory = MemberCategory::where('slug', $categorySlug)->first();
		
		if (!$parentCategory) {
			return [];
		}
		
		// Start with the parent category ID
		$categoryIds = [(int) $parentCategory->id];
		
		// Get all child categories of this parent
		$childCategories = MemberCategory::where('parent_id', $parentCategory->id)
			->pluck('id')
			->toArray();
		
		// Merge parent and children IDs
		if (!empty($childCategories)) {
			$categoryIds = array_merge($categoryIds, array_map('intval', $childCategories));
		}
		
		return array_unique($categoryIds);
	}

    /**
     * Render the component
     */
    public function render()
    {
        return view('livewire.frontend.widgets.top-members');
    }
}
