<?php

namespace App\Livewire\Member\Tabs;

use Livewire\Component;
use App\Models\User\User;
use App\Models\Blog\Post;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Filament\Tables\Concerns\InteractsWithTable;
use Filament\Tables\Contracts\HasTable;
use App\Enums\PostStatus;
use Filament\Tables\Actions\Action;
use Filament\Tables\Actions\ActionGroup;
use Filament\Tables\Actions\EditAction;
use Filament\Tables\Actions\DeleteAction;
use Filament\Tables\Actions\ForceDeleteAction;
use Filament\Tables\Actions\RestoreAction;
use Filament\Tables\Actions\ReplicateAction;
use Filament\Tables\Actions\BulkActionGroup;
use Filament\Tables\Actions\DeleteBulkAction;
use Filament\Tables\Actions\ForceDeleteBulkAction;
use Filament\Tables\Actions\RestoreBulkAction;
use Filament\Support\Enums\MaxWidth;
use Filament\Notifications\Notification;
use Filament\Support\Contracts\TranslatableContentDriver;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Concerns\InteractsWithForms;

class ArticlesTab extends Component implements HasTable, HasForms
{
    use InteractsWithTable;
    use InteractsWithForms;

    public User $member;

    public function mount(User $member)
    {
        $this->member = $member;
    }

    public function makeFilamentTranslatableContentDriver(): ?TranslatableContentDriver
    {
        return null;
    }

    public function table(Table $table): Table
    {
        return $table
            ->query($this->getTableQuery())
            ->columns([
                Tables\Columns\SpatieMediaLibraryImageColumn::make('featured_image')
                    ->collection('featured_image')
                    ->conversion('thumb')
                    ->size(40)
                    ->label('Image')
                    ->square(),
                Tables\Columns\TextColumn::make('post_title')
                    ->label('Title')
                    ->searchable()
                    ->tooltip(fn(Post $record): string => $record->post_title)
                    ->wrap()
                    ->formatStateUsing(fn(Post $record): string => \Illuminate\Support\Str::limit($record->post_title, 50)),
                Tables\Columns\TextColumn::make('categories.name')
                    ->wrap(),
                Tables\Columns\IconColumn::make('is_featured')
                    ->label('Featured'),
                Tables\Columns\TextColumn::make('post_status')
                    ->badge()
                    ->label('Status'),
                Tables\Columns\TextColumn::make('created_at')
                    ->label('Date')
                    ->dateTime()
                    ->wrap()
                    ->sortable()
                    ->getStateUsing(function (?Post $record): ?string {
                        if (!$record) return null;

                        return $record->post_status->value === PostStatus::Published->value
                            ? $record->published_at
                            : $record->created_at;
                    }),
            ])
            ->filters([
                Tables\Filters\TrashedFilter::make(),
                Tables\Filters\SelectFilter::make('categories')
                    ->relationship(
                        name: 'Categories',
                        titleAttribute: 'name',
                        modifyQueryUsing: fn(Builder $query) => $query
                            ->select(['post_categories.id', 'post_categories.name'])
                    )
                    ->multiple()
                    ->preload()
                    ->searchable()
                    ->optionsLimit(100)
                    ->placeholder('All'),
                Tables\Filters\TernaryFilter::make('is_featured')
                    ->label('Featured Posts'),
                Tables\Filters\SelectFilter::make('created_at')
                    ->label('Date')
                    ->options(function () {
                        $dates = Post::selectRaw('DATE_FORMAT(created_at, "%Y-%m") as month')
                            ->distinct()
                            ->pluck('month')
                            ->map(function ($date) {
                                return [
                                    $date => date('F Y', strtotime($date . '-01'))
                                ];
                            })
                            ->collapse()
                            ->toArray();

                        return $dates;
                    })
                    ->query(function (Builder $query, array $data): Builder {
                        return $query->when(
                            $data['value'],
                            fn(Builder $query, $date): Builder => $query->whereRaw('DATE_FORMAT(created_at, "%Y-%m") = ?', [$date])
                        );
                    }),
            ])
            ->filtersFormColumns(2)
            ->filtersFormWidth(MaxWidth::Medium)
            ->filtersTriggerAction(
                fn(Action $action) => $action
                    ->button()
                    ->label('Filters'),
            )
            ->actions([
                ActionGroup::make([
                    Action::make('preview')
                        ->icon('heroicon-o-eye')
                        ->color('info')
                        ->label(
                            fn(Post $record): string =>
                            $record->post_status === PostStatus::Published
                                ? 'View'
                                : 'Preview'
                        )
                        ->url(fn(Post $record): string => route('journal.show', [
                            'slug' => $record->slug,
                            'preview' => $record->post_status !== PostStatus::Published ? true : null,
                        ]), true)
                        ->hidden(
                            fn(Post $record): bool =>
                            !in_array($record->post_status->value, [
                                PostStatus::Published->value,
                                PostStatus::Scheduled->value,
                                PostStatus::Pending->value,
                                PostStatus::Draft->value,
                            ])
                        ),
                    Action::make('approve')
                        ->icon('heroicon-o-check-circle')
                        ->color('success')
                        ->visible(
                            fn(Post $record): bool =>
                            $record->post_status->value === PostStatus::Pending->value
                        )
                        ->action(function (Post $record): void {
                            $record->post_status = PostStatus::Published->value;
                            $record->published_at = now();
                            $record->save();
                        })
                        ->requiresConfirmation()
                        ->modalHeading('Approve Post')
                        ->modalDescription('Are you sure you want to approve this post? This will make it publicly visible.')
                        ->modalSubmitActionLabel('Yes, approve post')
                        ->successNotification(
                            Notification::make()
                                ->success()
                                ->title('Post Approved')
                                ->body('Post has been approved successfully')
                        ),
                    Action::make('publish')
                        ->icon('heroicon-o-arrow-up-circle')
                        ->color('success')
                        ->visible(
                            fn(Post $record): bool =>
                            in_array($record->post_status->value, [PostStatus::Draft->value, PostStatus::Expired->value])
                        )
                        ->action(function (Post $record): void {
                            $record->post_status = PostStatus::Published->value;
                            $record->published_at = now();
                            $record->save();
                        })
                        ->requiresConfirmation()
                        ->modalHeading('Publish Post')
                        ->modalDescription('Are you sure you want to publish this post? This will make it publicly visible.')
                        ->modalSubmitActionLabel('Yes, Publish Post')
                        ->successNotification(
                            Notification::make()
                                ->success()
                                ->title('Post Published')
                                ->body('Post has been published successfully')
                        ),

                    EditAction::make()
                        ->url(fn(Post $record): string => route('filament.espadmin.resources.blog.posts.edit', ['record' => $record->id]))
                        ->openUrlInNewTab()
                        ->hidden(fn(Post $record): bool => $record->trashed()),
                    ReplicateAction::make()
                        ->hidden(fn(Post $record): bool => $record->trashed())
                        ->beforeReplicaSaved(function (Post $replica): void {
                            $replica->post_status = PostStatus::Draft->value;
                            $replica->is_featured = false;
                        })
                        ->after(function (Post $replica, Post $record): void {
                            $replica->categories()->sync($record->categories->pluck('id'));
                            $replica->tags()->sync($record->tags->pluck('id'));
                        }),
                    DeleteAction::make()
                        ->hidden(fn(Post $record): bool => $record->trashed()),
                    ForceDeleteAction::make(),
                    RestoreAction::make(),
                ])->tooltip('Actions'),
            ])
            ->bulkActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                    ForceDeleteBulkAction::make(),
                    RestoreBulkAction::make(),
                ]),
            ])
            ->paginated([10, 25, 50, 100])
            ->defaultPaginationPageOption(10)
            ->deferLoading()
            ->defaultSort('created_at', 'desc');
    }

    public function getTableQuery(): Builder
    {
        return $this->member->articles()->getQuery();
    }

    public function render()
    {
        return view('livewire.member.tabs.articles-tab');
    }
}
