<?php

namespace App\Livewire\Member\Tabs;

use Livewire\Component;
use App\Models\User\User;
use App\Models\Classified\Classified;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Filament\Tables\Concerns\InteractsWithTable;
use Filament\Tables\Contracts\HasTable;
use App\Enums\ClassifiedStatus;
use Filament\Support\Enums\MaxWidth;
use Filament\Tables\Actions\Action;
use Filament\Notifications\Notification;
use Filament\Tables\Columns\SpatieMediaLibraryImageColumn;
use Filament\Support\Contracts\TranslatableContentDriver;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Concerns\InteractsWithForms;
use Illuminate\Support\Str;

class ClassifiedsTab extends Component implements HasTable, HasForms
{
    use InteractsWithTable;
    use InteractsWithForms;

    public User $member;

    public function mount(User $member)
    {
        $this->member = $member;
    }

    public function makeFilamentTranslatableContentDriver(): ?TranslatableContentDriver
    {
        return null;
    }

    public function table(Table $table): Table
    {
        return $table
            ->query($this->getTableQuery())
            ->columns([
                SpatieMediaLibraryImageColumn::make('featured_image')
                    ->collection('featured_image')
                    ->conversion('thumb')
                    ->size(40)
                    ->label('Image')
                    ->square(),
                Tables\Columns\TextColumn::make('title')
                    ->label('Title')
                    ->tooltip(fn(?Classified $record): string => $record?->title ?? '')
                    ->wrap()
                    ->formatStateUsing(fn(?Classified $record): string => $record ? Str::limit($record->title, 40) : ''),
                Tables\Columns\TextColumn::make('price')
                    ->money()
                    ->sortable(),
                Tables\Columns\TextColumn::make('categories.classified_category_name')
                    ->wrap(),
                Tables\Columns\IconColumn::make('is_featured')
                    ->label('Featured')
                    ->boolean()
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->alignCenter(),
                Tables\Columns\TextColumn::make('classified_status')
                    ->badge()
                    ->label('Status'),
                Tables\Columns\TextColumn::make('city.city_name')
                    ->formatStateUsing(fn(?Classified $record): string => 
                        $record ? collect([
                            $record->city?->city_name,
                            $record->state?->state_name,
                            $record->country?->country_name,
                            $record->postal_code
                        ])->filter()->join(', ') : ''
                    )
                    ->label('Location')
                    ->sortable()
                    ->wrap()
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->searchable(),
                Tables\Columns\TextColumn::make('published_at')
                    ->label('Date')
                    ->dateTime()
                    ->wrap()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: false)
                    ->getStateUsing(function (?Classified $record): ?string {
                        if (!$record) return null;
                        
                        return $record->classified_status->value === ClassifiedStatus::Published->value
                            ? $record->published_at
                            : $record->created_at;
                    }),
                Tables\Columns\TextColumn::make('expired_at')
                    ->dateTime()
                    ->wrap()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([    
                Tables\Filters\TrashedFilter::make(),
                Tables\Filters\TernaryFilter::make('is_featured')
                    ->label('Featured Classifieds'),
                Tables\Filters\SelectFilter::make('classified_categories')
                    ->label('Categories')
                    ->relationship(
                        name: 'Categories',
                        titleAttribute: 'classified_category_name',
                        modifyQueryUsing: fn(Builder $query) => $query
                            ->select(['classified_categories.id', 'classified_categories.classified_category_name'])
                    )
                    ->multiple()
                    ->preload()
                    ->searchable()
                    ->optionsLimit(100)
                    ->placeholder('All'),
                Tables\Filters\SelectFilter::make('classified_types')
                    ->label('Types')
                    ->relationship(
                        name: 'Types',
                        titleAttribute: 'classified_type_name',
                        modifyQueryUsing: fn(Builder $query) => $query
                            ->select(['classified_types.id', 'classified_types.classified_type_name'])
                    )
                    ->multiple()
                    ->preload()
                    ->searchable()
                    ->optionsLimit(100)
                    ->placeholder('All'),
                Tables\Filters\SelectFilter::make('state_id')
                    ->label('State')
                    ->relationship('state', 'state_name')
                    ->preload()
                    ->searchable()
                    ->optionsLimit(100)
                    ->placeholder('All'),
                Tables\Filters\SelectFilter::make('city_id')
                    ->label('City')
                    ->relationship('city', 'city_name')
                    ->preload()
                    ->searchable()
                    ->optionsLimit(100)
                    ->placeholder('All'),
                
                Tables\Filters\SelectFilter::make('created_at')
                    ->label('Date')
                    ->options(function () {
                        $dates = Classified::selectRaw('DATE_FORMAT(created_at, "%Y-%m") as month')
                            ->distinct()
                            ->pluck('month')
                            ->map(function ($date) {
                                return [
                                    $date => date('F Y', strtotime($date . '-01'))
                                ];
                            })
                            ->collapse()
                            ->toArray();

                        return $dates;
                    })
                    ->query(function (Builder $query, array $data): Builder {
                        return $query->when(
                            $data['value'],
                            fn(Builder $query, $date): Builder => $query->whereRaw('DATE_FORMAT(created_at, "%Y-%m") = ?', [$date])
                        );
                    }),
            ])
            ->filtersFormColumns(2)
            ->filtersFormWidth(MaxWidth::Medium)
            ->filtersTriggerAction(
                fn(Action $action) => $action
                    ->button()
                    ->label('Filter'),
            )
            ->actions([
                Tables\Actions\ActionGroup::make([
                    Tables\Actions\Action::make('preview')
                        ->icon('heroicon-o-eye')
                        ->color('info')
                        ->label(fn (Classified $record): string => 
                            $record->classified_status === ClassifiedStatus::Published 
                                ? 'View' 
                                : 'Preview'
                        )
                        ->url(fn (Classified $record): string => route('classified.show', [
                            'slug' => $record->slug,
                            'preview' => $record->classified_status !== ClassifiedStatus::Published ? true : null,
                        ]), true)
                        ->hidden(fn(Classified $record): bool => 
                            !in_array($record->classified_status->value, [
                                ClassifiedStatus::Published->value,
                                ClassifiedStatus::Scheduled->value,
                                ClassifiedStatus::Pending->value,
                                ClassifiedStatus::Draft->value,
                            ])
                        ),
                    Tables\Actions\Action::make('approve')
                        ->icon('heroicon-o-check-circle')
                        ->color('success')
                        ->visible(fn (Classified $record): bool => 
                            $record->classified_status?->value === ClassifiedStatus::Pending->value
                        )
                        ->action(function (Classified $record): void {
                            $record->classified_status = ClassifiedStatus::Published->value;
                            $record->published_at = now();
                            $record->save();
                        })
                        ->requiresConfirmation()
                        ->modalHeading('Approve Classified')
                        ->modalDescription('Are you sure you want to approve this classified? This will make it publicly visible.')
                        ->modalSubmitActionLabel('Yes, approve classified')
                        ->successNotification(
                            Notification::make()
                                ->success()
                                ->title('Classified approved')
                                ->body('Classified has been approved successfully')
                        ),
                    Tables\Actions\Action::make('publish')
                        ->icon('heroicon-o-arrow-up-circle')
                        ->color('success')
                        ->visible(fn (Classified $record): bool => 
                            in_array($record->classified_status->value, [ClassifiedStatus::Draft->value, ClassifiedStatus::Expired->value])
                        )
                        ->action(function (Classified $record): void {
                            $record->classified_status = ClassifiedStatus::Published->value;
                            $record->published_at = now();
                            $record->save();
                        })
                        ->requiresConfirmation()
                        ->modalHeading('Publish Classified')
                        ->modalDescription('Are you sure you want to publish this classified? This will make it publicly visible.')
                        ->modalSubmitActionLabel('Yes, Publish Classified')
                        ->successNotification(
                            Notification::make()
                                ->success()
                                ->title('Classified published')
                                ->body('Classified has been published successfully')
                        ),
                    Tables\Actions\Action::make('toggleFeatured')
                        ->icon(fn (Classified $record): string => $record->is_featured ? 'heroicon-o-star' : 'heroicon-o-star')
                        ->color(fn (Classified $record): string => $record->is_featured ? 'warning' : 'gray')
                        ->label(fn (Classified $record): string => $record->is_featured ? 'Unset Featured' : 'Set Featured')
                        ->visible(fn (Classified $record): bool => 
                            $record->classified_status->value === ClassifiedStatus::Published->value
                        )
                        ->action(function (Classified $record): void {
                            $record->is_featured = !$record->is_featured;
                            $record->save();
                        })
                        ->requiresConfirmation()
                        ->modalHeading(fn (Classified $record): string => 
                            $record->is_featured ? 'Unset Featured Classified' : 'Set as Featured Classified'
                        )
                        ->modalDescription(fn (Classified $record): string => 
                            $record->is_featured 
                                ? 'Are you sure you want to unset this classified as featured?' 
                                : 'Are you sure you want to set this classified as featured?'
                        )
                        ->modalSubmitActionLabel(fn (Classified $record): string => 
                            $record->is_featured ? 'Yes, Unset Featured' : 'Yes, Set as Featured'
                        )
                        ->successNotification(
                            Notification::make()
                                ->success()
                                ->title(fn (Classified $record): string => 
                                    $record->is_featured ? 'Classified set as featured' : 'Classified unset as featured'
                                )
                                ->body(fn (Classified $record): string => 
                                    $record->is_featured 
                                        ? 'Classified has been set as featured successfully' 
                                        : 'Classified has been unset as featured successfully'
                                )
                        ),
                    Tables\Actions\EditAction::make() 
                        ->url(fn (Classified $record): string => route('filament.espadmin.resources.classified.classifieds.edit', ['record' => $record->id]))
                        ->openUrlInNewTab()
                        ->hidden(fn(Classified $record): bool => $record->trashed()),
                    Tables\Actions\ReplicateAction::make()
                        ->hidden(fn(Classified $record): bool => $record->trashed())
                        ->beforeReplicaSaved(function (Classified $replica): void {
                            $replica->classified_status = ClassifiedStatus::Draft->value;
                            $replica->is_featured = false;
                        })
                        ->after(function (Classified $replica, Classified $record): void {
                            $replica->categories()->sync($record->categories->pluck('id'));
                            $replica->types()->sync($record->types->pluck('id'));
                            $replica->tags()->sync($record->tags->pluck('id'));
                        }),
                    Tables\Actions\DeleteAction::make()
                        ->hidden(fn(Classified $record): bool => $record->trashed()),
                    Tables\Actions\ForceDeleteAction::make(),
                    Tables\Actions\RestoreAction::make(),
                ])->tooltip('Actions'),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\ForceDeleteBulkAction::make(),
                    Tables\Actions\RestoreBulkAction::make(),
                ]),
            ])
            ->paginated([10, 25, 50, 100])
            ->defaultPaginationPageOption(10)
            ->deferLoading()
            ->defaultSort('created_at', 'desc')
            ->defaultSort('published_at', 'desc');
    }

    public function getTableQuery(): Builder
    {
        return $this->member->classifieds()->getQuery();
    }

    public function render()
    {
        return view('livewire.member.tabs.classifieds-tab');
    }
} 