<?php

namespace App\Livewire\Member\Tabs;

use Livewire\Component;
use App\Models\User\User;
use App\Models\Membership\Payment;
use App\Enums\PaymentType;
use App\Enums\PaymentMethod;
use App\Enums\PaymentStatus;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Filament\Tables\Concerns\InteractsWithTable;
use Filament\Tables\Contracts\HasTable;
use Filament\Support\Enums\MaxWidth;
use Filament\Tables\Actions\Action;
use Filament\Notifications\Notification;
use Filament\Support\Contracts\TranslatableContentDriver;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Actions\ViewAction;
use Filament\Tables\Actions\ActionGroup;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\DatePicker;
use Illuminate\Support\Str;

class InvoicesTab extends Component implements HasTable, HasForms
{
    use InteractsWithTable;
    use InteractsWithForms;

    public User $member;

    public function mount(User $member): void
    {
        $this->member = $member;
    }

    public function makeFilamentTranslatableContentDriver(): ?TranslatableContentDriver
    {
        return null;
    }

    public function table(Table $table): Table
    {
        return $table
            ->query($this->getTableQuery())
            ->columns([
                Tables\Columns\TextColumn::make('payment_date')
                    ->label('Date')
                    ->dateTime()
                    ->sortable()
                    ->since()
                    ->toggleable(),
                Tables\Columns\TextColumn::make('transaction_id')
                    ->label('Transaction ID')
                    ->searchable()
                    ->copyable()
                    ->formatStateUsing(fn($state) => Str::limit($state, 15))
                    ->tooltip(function (TextColumn $column): ?string {
                        $state = $column->getState();
                        if (strlen($state) <= 15) {
                            return null;
                        }
                        return $state;
                    }),
                Tables\Columns\TextColumn::make('payment_type')
                    ->label('Type')
                    ->badge()
                    ->searchable(),
                Tables\Columns\TextColumn::make('payment_method')
                    ->label('Method')
                    ->badge()
                    ->searchable(),
                Tables\Columns\TextColumn::make('amount')
                    ->label('Amount')
                    ->money('USD')
                    ->sortable(),
                Tables\Columns\TextColumn::make('status')
                    ->label('Status')
                    ->badge()
                    ->searchable(),
                Tables\Columns\TextColumn::make('purchase.item_type')
                    ->label('Purchase Type')
                    ->badge()
                    ->toggleable(),
                Tables\Columns\TextColumn::make('purchase_details')
                    ->label('Purchase Item')
                    ->getStateUsing(function (Payment $record) {
                        $purchase = $record->purchase;
                        if (!$purchase) return 'N/A';

                        if ($purchase->purchaseable_type === 'App\\Models\\Membership\\Subscription') {
                            $subscription = $purchase->purchaseable;
                            return $subscription?->membershipPlan?->package_title ?? 'Unknown Plan';
                        }
                        return $purchase->purchaseable_type ? class_basename($purchase->purchaseable_type) : 'N/A';
                    })
                    ->wrap()
                    ->toggleable(),
                Tables\Columns\TextColumn::make('gateway_fee')
                    ->label('Gateway Fee')
                    ->getStateUsing(function (Payment $record) {
                        $response = $record->service_response ?? [];
                        return isset($response['fee']) ? '$' . number_format((float)$response['fee'], 2) : 'N/A';
                    })
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('notes')
                    ->label('Notes')
                    ->limit(30)
                    ->tooltip(function (TextColumn $column): ?string {
                        $state = $column->getState();
                        if (strlen($state) <= 30) {
                            return null;
                        }
                        return $state;
                    })
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Tables\Filters\SelectFilter::make('payment_type')
                    ->label('Payment Type')
                    ->options([
                        PaymentType::Payment->value => PaymentType::Payment->getLabel(),
                        PaymentType::Refund->value => PaymentType::Refund->getLabel(),
                    ]),
                Tables\Filters\SelectFilter::make('payment_method')
                    ->label('Payment Method')
                    ->options([
                        PaymentMethod::Stripe->value => PaymentMethod::Stripe->getLabel(),
                        PaymentMethod::Paypal->value => PaymentMethod::Paypal->getLabel(),
                    ]),
                Tables\Filters\SelectFilter::make('status')
                    ->label('Status')
                    ->options([
                        PaymentStatus::Paid->value => PaymentStatus::Paid->getLabel(),
                        PaymentStatus::Pending->value => PaymentStatus::Pending->getLabel(),
                        PaymentStatus::Failed->value => PaymentStatus::Failed->getLabel(),
                        PaymentStatus::Refunded->value => PaymentStatus::Refunded->getLabel(),
                    ]),
                Tables\Filters\Filter::make('amount_range')
                    ->form([
                        TextInput::make('amount_from')
                            ->label('Amount From')
                            ->numeric(),
                        TextInput::make('amount_to')
                            ->label('Amount To')
                            ->numeric(),
                    ])
                    ->query(function (Builder $query, array $data): Builder {
                        return $query
                            ->when(
                                $data['amount_from'],
                                fn(Builder $query, $amount): Builder => $query->where('amount', '>=', $amount)
                            )
                            ->when(
                                $data['amount_to'],
                                fn(Builder $query, $amount): Builder => $query->where('amount', '<=', $amount)
                            );
                    }),
                Tables\Filters\Filter::make('date_range')
                    ->form([
                        DatePicker::make('date_from')
                            ->label('Date From'),
                        DatePicker::make('date_to')
                            ->label('Date To'),
                    ])
                    ->query(function (Builder $query, array $data): Builder {
                        return $query
                            ->when(
                                $data['date_from'],
                                fn(Builder $query, $date): Builder => $query->whereDate('payment_date', '>=', $date)
                            )
                            ->when(
                                $data['date_to'],
                                fn(Builder $query, $date): Builder => $query->whereDate('payment_date', '<=', $date)
                            );
                    }),
            ])
            ->filtersFormColumns(2)
            ->filtersFormWidth(MaxWidth::Medium)
            ->actions([
                ActionGroup::make([
                    ViewAction::make()
                        ->label('View Details')
                        ->url(fn(Payment $record): string => route('filament.espadmin.resources.membership.invoices.view', $record->purchase))
                        ->openUrlInNewTab(),

                    Action::make('download_receipt')
                        ->label('Download Invoice')
                        ->icon('heroicon-o-document-arrow-down')
                        ->color('success')
                        ->visible(fn(Payment $record): bool => $record->isSuccessful())
                        ->url(fn(Payment $record): string => route('dashboard.invoice.generate', $record->uuid))
                        ->openUrlInNewTab(),

                ])->tooltip('Actions'),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\BulkAction::make('export')
                        ->label('Export Selected')
                        ->icon('heroicon-o-document-arrow-down')
                        ->action(function (\Illuminate\Database\Eloquent\Collection $records): void {
                            Notification::make()
                                ->success()
                                ->title('Export Started')
                                ->body('Export functionality would be implemented here')
                                ->send();
                        }),
                    Tables\Actions\BulkAction::make('download_receipts')
                        ->label('Download Receipts')
                        ->icon('heroicon-o-document-duplicate')
                        ->action(function (\Illuminate\Database\Eloquent\Collection $records): void {
                            $successfulPayments = $records->filter(fn(Payment $payment) => $payment->isSuccessful());
                            Notification::make()
                                ->success()
                                ->title('Receipts Download')
                                ->body("Download {$successfulPayments->count()} receipts functionality would be implemented here")
                                ->send();
                        }),
                ]),
            ])
            ->paginated([10, 25, 50, 100])
            ->defaultPaginationPageOption(10)
            ->deferLoading()
            ->defaultSort('payment_date', 'desc')
            ->emptyStateHeading('No payments found')
            ->emptyStateDescription('This member has not made any payments yet.')
            ->emptyStateIcon('heroicon-o-credit-card');
    }

    public function getTableQuery(): Builder
    {
        return $this->member->payments()
            ->with(['purchase.purchaseable'])
            ->getQuery();
    }

    public function render()
    {
        return view('livewire.member.tabs.invoices-tab');
    }
}
