<?php

namespace App\Livewire\Member\Tabs;

use Livewire\Component;
use App\Models\User\User;
use App\Models\Job\Job;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Filament\Tables\Concerns\InteractsWithTable;
use Filament\Tables\Contracts\HasTable;
use App\Enums\JobStatus;
use Filament\Tables\Actions\Action;
use Filament\Support\Enums\MaxWidth;
use Filament\Notifications\Notification;
use Filament\Support\Contracts\TranslatableContentDriver;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Concerns\InteractsWithForms;
use Illuminate\Support\Str;
use Carbon\Carbon;
use Filament\Forms;
use App\Enums\JobShift;


class JobsTab extends Component implements HasTable, HasForms
{
    use InteractsWithTable;
    use InteractsWithForms;

    public User $member;

    public function mount(User $member)
    {
        $this->member = $member;
    }

    public function makeFilamentTranslatableContentDriver(): ?TranslatableContentDriver
    {
        return null;
    }

    public function table(Table $table): Table
    {
        return $table
            ->query($this->getTableQuery())
            ->columns([
                Tables\Columns\TextColumn::make('job_title')
                    ->formatStateUsing(fn($state) => Str::limit($state, 30))
                    ->searchable(),
                Tables\Columns\TextColumn::make('job_role.job_role_name')
                    ->sortable(),
                Tables\Columns\TextColumn::make('job_status')
                    ->badge()
                    ->toggleable(isToggledHiddenByDefault: false)
                    ->label('Status'),
                Tables\Columns\IconColumn::make('is_featured')
                    ->label('Featured')
                    ->toggleable(isToggledHiddenByDefault: false),
                Tables\Columns\TextColumn::make('salary_period')
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->searchable(),
                Tables\Columns\TextColumn::make('job_shift')
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->searchable(),
                Tables\Columns\TextColumn::make('apply_by_date')
                    ->date()
                    ->sortable(),
                Tables\Columns\TextColumn::make('city.city_name')
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->sortable(),
                Tables\Columns\TextColumn::make('dates')
                    ->label('Dates')
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->view('filament.resources.common.tables.columns.dates'),
            ])
            ->filters([
                Tables\Filters\TrashedFilter::make(),
                Tables\Filters\SelectFilter::make('job_role')
                    ->relationship('job_role', 'job_role_name')
                    ->searchable()
                    ->preload()
                    ->label('Job Role'),
                Tables\Filters\SelectFilter::make('job_status')
                    ->options(JobStatus::class)
                    ->label('Job Status'),
                Tables\Filters\SelectFilter::make('job_type')
                    ->relationship('job_type', 'job_type_name')
                    ->searchable()
                    ->preload()
                    ->label('Job Type'),
                Tables\Filters\SelectFilter::make('job_shift')
                    ->options(JobShift::class)
                    ->label('Job Shift'),
                Tables\Filters\TernaryFilter::make('is_featured')
                    ->label('Featured Status')
                    ->placeholder('All Jobs')
                    ->trueLabel('Featured Jobs')
                    ->falseLabel('Non-Featured Jobs'),
                Tables\Filters\Filter::make('published_at')
                    ->form([
                        Forms\Components\DatePicker::make('published_from')
                            ->label('Published From'),
                        Forms\Components\DatePicker::make('published_until')
                            ->label('Published Until'),
                    ])
                    ->query(function (Builder $query, array $data): Builder {
                        return $query
                            ->when(
                                $data['published_from'],
                                fn(Builder $query, $date): Builder => $query->whereDate('published_at', '>=', $date),
                            )
                            ->when(
                                $data['published_until'],
                                fn(Builder $query, $date): Builder => $query->whereDate('published_at', '<=', $date),
                            );
                    })
                    ->indicateUsing(function (array $data): array {
                        $indicators = [];
                        if ($data['published_from'] ?? null) {
                            $indicators['published_from'] = 'Published from ' . Carbon::parse($data['published_from'])->toFormattedDateString();
                        }
                        if ($data['published_until'] ?? null) {
                            $indicators['published_until'] = 'Published until ' . Carbon::parse($data['published_until'])->toFormattedDateString();
                        }
                        return $indicators;
                    }),
                Tables\Filters\Filter::make('expired_at')
                    ->form([
                        Forms\Components\DatePicker::make('expired_from')
                            ->label('Expires From'),
                        Forms\Components\DatePicker::make('expired_until')
                            ->label('Expires Until'),
                    ])
                    ->query(function (Builder $query, array $data): Builder {
                        return $query
                            ->when(
                                $data['expired_from'],
                                fn(Builder $query, $date): Builder => $query->whereDate('expired_at', '>=', $date),
                            )
                            ->when(
                                $data['expired_until'],
                                fn(Builder $query, $date): Builder => $query->whereDate('expired_at', '<=', $date),
                            );
                    })
                    ->indicateUsing(function (array $data): array {
                        $indicators = [];
                        if ($data['expired_from'] ?? null) {
                            $indicators['expired_from'] = 'Expires from ' . Carbon::parse($data['expired_from'])->toFormattedDateString();
                        }
                        if ($data['expired_until'] ?? null) {
                            $indicators['expired_until'] = 'Expires until ' . Carbon::parse($data['expired_until'])->toFormattedDateString();
                        }
                        return $indicators;
                    }),
                Tables\Filters\SelectFilter::make('state')
                    ->relationship('state', 'state_name')
                    ->searchable()
                    ->preload()
                    ->label('State'),
                Tables\Filters\SelectFilter::make('city')
                    ->relationship('city', 'city_name')
                    ->searchable()
                    ->preload()
                    ->label('City'),
            ])
            ->filtersFormColumns(2)
            ->filtersFormWidth(MaxWidth::Medium)
            ->filtersTriggerAction(
                fn(Action $action) => $action
                    ->button()
                    ->label('Filters'),
            )
            ->actions([
                Tables\Actions\ActionGroup::make([
                    Tables\Actions\Action::make('preview')
                        ->icon('heroicon-o-eye')
                        ->color('info')
                        ->label(
                            fn(Job $record): string =>
                            $record->job_status === JobStatus::Published
                                ? 'View'
                                : 'Preview'
                        )
                        ->url(fn(Job $record): string => route('job.show', [
                            'slug' => $record->slug,
                            'preview' => $record->job_status !== JobStatus::Published ? true : null,
                        ]), true)
                        ->hidden(
                            fn(Job $record): bool =>
                            !in_array($record->job_status->value, [
                                JobStatus::Published->value,
                                JobStatus::Scheduled->value,
                                JobStatus::Pending->value,
                                JobStatus::Draft->value,
                            ])
                        ),
                    Tables\Actions\Action::make('approve')
                        ->icon('heroicon-o-check-circle')
                        ->color('success')
                        ->visible(
                            fn(Job $record): bool =>
                            $record->job_status?->value === JobStatus::Pending->value
                        )
                        ->action(function (Job $record): void {
                            $record->job_status = JobStatus::Published->value;
                            $record->published_at = now();
                            $record->save();
                        })
                        ->requiresConfirmation()
                        ->modalHeading('Approve Job')
                        ->modalDescription('Are you sure you want to approve this job? This will make it publicly visible.')
                        ->modalSubmitActionLabel('Yes, approve job')
                        ->successNotification(
                            Notification::make()
                                ->success()
                                ->title('Job approved')
                                ->body('Job has been approved successfully')
                        ),
                    Tables\Actions\Action::make('publish')
                        ->icon('heroicon-o-arrow-up-circle')
                        ->color('success')
                        ->visible(
                            fn(Job $record): bool =>
                            in_array($record->job_status->value, [JobStatus::Draft->value, JobStatus::Expired->value])
                        )
                        ->action(function (Job $record): void {
                            $record->job_status = JobStatus::Published->value;
                            $record->published_at = now();
                            $record->save();
                        })
                        ->requiresConfirmation()
                        ->modalHeading('Publish Classified')
                        ->modalDescription('Are you sure you want to publish this job? This will make it publicly visible.')
                        ->modalSubmitActionLabel('Yes, Publish Job')
                        ->successNotification(
                            Notification::make()
                                ->success()
                                ->title('Job published')
                                ->body('Job has been published successfully')
                        ),
                    Tables\Actions\Action::make('toggleFeatured')
                        ->icon(fn(Job $record): string => $record->is_featured ? 'heroicon-o-star' : 'heroicon-o-star')
                        ->color(fn(Job $record): string => $record->is_featured ? 'warning' : 'gray')
                        ->label(fn(Job $record): string => $record->is_featured ? 'Unset Featured' : 'Set Featured')
                        ->visible(
                            fn(Job $record): bool =>
                            $record->job_status->value === JobStatus::Published->value
                        )
                        ->action(function (Job $record): void {
                            $record->is_featured = !$record->is_featured;
                            $record->save();
                        })
                        ->requiresConfirmation()
                        ->modalHeading(
                            fn(Job $record): string =>
                            $record->is_featured ? 'Unset Featured Job' : 'Set as Featured Job'
                        )
                        ->modalDescription(
                            fn(Job $record): string =>
                            $record->is_featured
                                ? 'Are you sure you want to unset this classified as featured?'
                                : 'Are you sure you want to set this classified as featured?'
                        )
                        ->modalSubmitActionLabel(
                            fn(Job $record): string =>
                            $record->is_featured ? 'Yes, Unset Featured' : 'Yes, Set as Featured'
                        )
                        ->successNotification(
                            Notification::make()
                                ->success()
                                ->title(
                                    fn(Job $record): string =>
                                    $record->is_featured ? 'Job set as featured' : 'Job unset as featured'
                                )
                                ->body(
                                    fn(Job $record): string =>
                                    $record->is_featured
                                        ? 'Job has been set as featured successfully'
                                        : 'Job has been unset as featured successfully'
                                )
                        ),
                    Tables\Actions\EditAction::make()
                        ->url(fn(Job $record): string => route('filament.espadmin.resources.job.jobs.edit', ['record' => $record->id]))
                        ->openUrlInNewTab()
                        ->hidden(fn(Job $record): bool => $record->trashed()),
                    Tables\Actions\ReplicateAction::make()
                        ->beforeReplicaSaved(function (Job $replica): void {
                            $replica->job_status = JobStatus::Draft->value;
                            $replica->is_featured = false;
                        })
                        ->after(function (Job $replica, Job $record): void {
                            $replica->job_skills()->sync($record->job_skills->pluck('id'));
                        })
                        ->hidden(fn(Job $record): bool => $record->trashed()),
                    Tables\Actions\DeleteAction::make()
                        ->hidden(fn(Job $record): bool => $record->trashed()),
                    Tables\Actions\ForceDeleteAction::make(),
                    Tables\Actions\RestoreAction::make(),
                ])->tooltip('Actions'),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\ForceDeleteBulkAction::make(),
                    Tables\Actions\RestoreBulkAction::make(),
                ]),
            ])
            ->paginated([10, 25, 50, 100])
            ->defaultPaginationPageOption(10)
            ->deferLoading()
            ->defaultSort('created_at', 'desc');
    }

    public function getTableQuery(): Builder
    {
        return $this->member->jobs()->getQuery();
    }

    public function render()
    {
        return view('livewire.member.tabs.jobs-tab');
    }
}
