<?php

namespace App\Livewire\Member\Tabs;

use Livewire\Component;
use App\Models\User\User;
use App\Models\Project\Project;
use App\Enums\ProjectStatus;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Filament\Tables\Concerns\InteractsWithTable;
use Filament\Tables\Contracts\HasTable;
use Filament\Support\Enums\MaxWidth;
use Filament\Tables\Actions\Action;
use Filament\Notifications\Notification;
use Filament\Tables\Columns\SpatieMediaLibraryImageColumn;
use Filament\Support\Contracts\TranslatableContentDriver;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Actions\ViewAction;
use Filament\Tables\Actions\EditAction;
use Filament\Tables\Actions\ReplicateAction;
use Filament\Tables\Actions\DeleteAction;
use Filament\Tables\Actions\ForceDeleteAction;
use Filament\Tables\Actions\RestoreAction;
use Filament\Tables\Actions\BulkActionGroup;
use Filament\Tables\Actions\DeleteBulkAction;
use Filament\Tables\Actions\ForceDeleteBulkAction;
use Filament\Tables\Actions\RestoreBulkAction;
use Filament\Tables\Actions\ActionGroup;
use Illuminate\Support\Str;

class ProjectsTab extends Component implements HasTable, HasForms
{
    use InteractsWithTable;
    use InteractsWithForms;

    public User $member;

    public function mount(User $member)
    {
        $this->member = $member;
    }

    public function makeFilamentTranslatableContentDriver(): ?TranslatableContentDriver
    {
        return null;
    }

    public function table(Table $table): Table
    {
        return $table
            ->query($this->getTableQuery())
            ->columns([
                SpatieMediaLibraryImageColumn::make('featured_image')
                    ->collection('featured_image')
                    ->conversion('thumb')
                    ->size(40)
                    ->label('Image')
                    ->square(),
                Tables\Columns\TextColumn::make('title')
                    ->searchable()
                    ->formatStateUsing(fn($state) => Str::limit($state, 20)),
                Tables\Columns\TextColumn::make('types.type_name')
                    ->label('Type')
                    ->wrap(),
                Tables\Columns\TextColumn::make('project_status')
                    ->label('Status')
                    ->badge()
                    ->searchable(),
                Tables\Columns\TextColumn::make('published_at')
                    ->dateTime()
                    ->sortable(),
                Tables\Columns\TextColumn::make('expired_at')
                    ->dateTime()
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->sortable(),
            ])
            ->filters([
                Tables\Filters\TrashedFilter::make(),
                Tables\Filters\SelectFilter::make('team_member_status')
                    ->options([
                        ProjectStatus::Draft->value => ProjectStatus::Draft->name,
                        ProjectStatus::Published->value => ProjectStatus::Published->name,
                        ProjectStatus::Pending->value => ProjectStatus::Pending->name,
                        ProjectStatus::Expired->value => ProjectStatus::Expired->name,
                        ProjectStatus::Trashed->value => ProjectStatus::Trashed->name,
                    ])
                    ->label('Status'),
            ])
            ->filtersFormColumns(2)
            ->filtersFormWidth(MaxWidth::Medium)
            ->actions([
                ActionGroup::make([

                    Action::make('approve')
                        ->icon('heroicon-o-check-circle')
                        ->color('success')
                        ->visible(
                            fn(Project $record): bool =>
                            $record->project_status->value === ProjectStatus::Pending->value
                        )
                        ->action(function (Project $record): void {
                            $record->project_status = ProjectStatus::Published->value;
                            $record->published_at = now();
                            $record->save();
                        })
                        ->requiresConfirmation()
                        ->modalHeading('Approve Project')
                        ->modalDescription('Are you sure you want to approve this project? This will make it publicly visible.')
                        ->modalSubmitActionLabel('Yes, approve')
                        ->successNotification(
                            Notification::make()
                                ->success()
                                ->title('Project Approved')
                                ->body('Project has been approved successfully')
                        ),
                    Action::make('publish')
                        ->icon('heroicon-o-arrow-up-circle')
                        ->color('success')
                        ->visible(
                            fn(Project $record): bool =>
                            in_array($record->project_status->value, [ProjectStatus::Draft->value, ProjectStatus::Expired->value])
                        )
                        ->action(function (Project $record): void {
                            $record->project_status = ProjectStatus::Published->value;
                            $record->published_at = now();
                            $record->save();
                        })
                        ->requiresConfirmation()
                        ->modalHeading('Publish Project')
                        ->modalDescription('Are you sure you want to publish this project? This will make it publicly visible.')
                        ->modalSubmitActionLabel('Yes, Publish')
                        ->successNotification(
                            Notification::make()
                                ->success()
                                ->title('Project Published')
                                ->body('Project has been published successfully')
                        ),

                    EditAction::make()
                        ->url(fn(Project $record): string => route('filament.espadmin.resources.project.projects.edit', ['record' => $record->id]))
                        ->openUrlInNewTab()
                        ->hidden(fn(Project $record): bool => $record->trashed()),
                    ReplicateAction::make()
                        ->before(function (Project $record): void {
                            $record->project_status = ProjectStatus::Draft->value;
                        })
                        ->after(function (Project $replica, Project $record): void {
                            $replica->types()->sync($record->types->pluck('id'));
                        })
                        ->hidden(fn(Project $record): bool => $record->trashed()),
                    DeleteAction::make()
                        ->hidden(fn(Project $record): bool => $record->trashed()),
                    ForceDeleteAction::make(),
                    RestoreAction::make(),
                ])->tooltip('Actions'),
            ])
            ->bulkActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                    ForceDeleteBulkAction::make(),
                    RestoreBulkAction::make(),
                ]),
            ])
            ->paginated([10, 25, 50, 100])
            ->defaultPaginationPageOption(10)
            ->deferLoading()
            ->defaultSort('created_at', 'desc');
    }

    public function getTableQuery(): Builder
    {

        return $this->member->projects()->getQuery();
    }

    public function render()
    {
        return view('livewire.member.tabs.projects-tab');
    }
}
