<?php

namespace App\Livewire\Member\Tabs;

use Livewire\Component;
use App\Models\User\User;
use App\Models\Review;
use App\Enums\MemberRatingType;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Filament\Tables\Concerns\InteractsWithTable;
use Filament\Tables\Contracts\HasTable;
use Filament\Support\Enums\MaxWidth;
use Filament\Tables\Actions\Action;
use Filament\Notifications\Notification;
use Filament\Support\Contracts\TranslatableContentDriver;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Actions\ViewAction;
use Filament\Tables\Actions\EditAction;
use Filament\Tables\Actions\DeleteAction;
use Filament\Tables\Actions\ForceDeleteAction;
use Filament\Tables\Actions\RestoreAction;
use Filament\Tables\Actions\BulkActionGroup;
use Filament\Tables\Actions\DeleteBulkAction;
use Filament\Tables\Actions\ForceDeleteBulkAction;
use Filament\Tables\Actions\RestoreBulkAction;
use Filament\Tables\Actions\ActionGroup;
use Illuminate\Support\Str;

class ReviewsTab extends Component implements HasTable, HasForms
{
    use InteractsWithTable;
    use InteractsWithForms;

    public User $member;

    public function mount(User $member): void
    {
        $this->member = $member;
    }

    public function makeFilamentTranslatableContentDriver(): ?TranslatableContentDriver
    {
        return null;
    }

    public function table(Table $table): Table
    {
        return $table
            ->query($this->getTableQuery())
            ->columns([
                Tables\Columns\TextColumn::make('user.name')
                    ->label('Reviewer')
                    ->searchable()
                    ->sortable()
                    ->formatStateUsing(fn($state) => Str::limit($state, 20)),
                Tables\Columns\TextColumn::make('review_title')
                    ->label('Title')
                    ->searchable()
                    ->formatStateUsing(fn($state) => Str::limit($state, 30))
                    ->wrap(),
                Tables\Columns\TextColumn::make('review_content')
                    ->label('Content')
                    ->formatStateUsing(fn($state) => Str::limit(strip_tags($state), 50))
                    ->wrap()
                    ->toggleable(),
                Tables\Columns\TextColumn::make('review_rating')
                    ->label('Overall Rating')
                    ->badge()
                    ->colors([
                        'danger' => fn($state) => $state <= 2,
                        'warning' => fn($state) => $state == 3,
                        'primary' => fn($state) => $state == 4,
                        'success' => fn($state) => $state == 5,
                    ])
                    ->formatStateUsing(fn($state) => $state . '/5 ⭐'),
                Tables\Columns\TextColumn::make('average_rating')
                    ->label('Detailed Rating')
                    ->getStateUsing(function (Review $record) {
                        return $record->average_rating ?? 0;
                    })
                    ->formatStateUsing(fn($state) => number_format($state, 1) . '/5 ⭐')
                    ->sortable()
                    ->toggleable(),
                Tables\Columns\IconColumn::make('status')
                    ->label('Status')
                    ->boolean()
                    ->trueIcon('heroicon-o-check-circle')
                    ->falseIcon('heroicon-o-x-circle')
                    ->trueColor('success')
                    ->falseColor('danger'),
                Tables\Columns\TextColumn::make('created_at')
                    ->label('Date')
                    ->dateTime()
                    ->sortable()
                    ->since()
                    ->toggleable(),
                Tables\Columns\TextColumn::make('user.email')
                    ->label('Reviewer Email')
                    ->searchable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Tables\Filters\TrashedFilter::make(),
                Tables\Filters\SelectFilter::make('review_rating')
                    ->label('Overall Rating')
                    ->options([
                        1 => '1 Star',
                        2 => '2 Stars',
                        3 => '3 Stars',
                        4 => '4 Stars',
                        5 => '5 Stars',
                    ]),
                Tables\Filters\TernaryFilter::make('status')
                    ->label('Status')
                    ->placeholder('All reviews')
                    ->trueLabel('Approved only')
                    ->falseLabel('Pending only'),
            ])
            ->filtersFormColumns(2)
            ->filtersFormWidth(MaxWidth::Medium)
            ->actions([
                ActionGroup::make([
                    Action::make('approve')
                        ->icon('heroicon-o-check-circle')
                        ->color('success')
                        ->visible(fn(Review $record): bool => !$record->status)
                        ->action(function (Review $record): void {
                            $record->status = true;
                            $record->save();
                        })
                        ->requiresConfirmation()
                        ->modalHeading('Approve Review')
                        ->modalDescription('Are you sure you want to approve this review? It will be visible to the public.')
                        ->modalSubmitActionLabel('Yes, approve')
                        ->successNotification(
                            Notification::make()
                                ->success()
                                ->title('Review Approved')
                                ->body('Review has been approved successfully')
                        ),
                    Action::make('disapprove')
                        ->icon('heroicon-o-x-circle')
                        ->color('warning')
                        ->visible(fn(Review $record): bool => $record->status)
                        ->action(function (Review $record): void {
                            $record->status = false;
                            $record->save();
                        })
                        ->requiresConfirmation()
                        ->modalHeading('Disapprove Review')
                        ->modalDescription('Are you sure you want to disapprove this review? It will be hidden from public view.')
                        ->modalSubmitActionLabel('Yes, disapprove')
                        ->successNotification(
                            Notification::make()
                                ->warning()
                                ->title('Review Disapproved')
                                ->body('Review has been disapproved successfully')
                        ),
                    ViewAction::make()
                        ->label('View in Admin')
                        ->openUrlInNewTab()
                        ->hidden(fn(Review $record): bool => $record->trashed()),
                    EditAction::make()
                        ->label('Edit in Admin')
                        ->openUrlInNewTab()
                        ->hidden(fn(Review $record): bool => $record->trashed()),
                    DeleteAction::make()
                        ->hidden(fn(Review $record): bool => $record->trashed()),
                    ForceDeleteAction::make(),
                    RestoreAction::make(),
                ])->tooltip('Actions'),
            ])
            ->bulkActions([
                BulkActionGroup::make([
                    Tables\Actions\BulkAction::make('approve')
                        ->label('Approve Selected')
                        ->icon('heroicon-o-check-circle')
                        ->color('success')
                        ->action(function (\Illuminate\Database\Eloquent\Collection $records): void {
                            $records->each(function (Review $record) {
                                $record->status = true;
                                $record->save();
                            });
                        })
                        ->requiresConfirmation()
                        ->modalHeading('Approve Selected Reviews')
                        ->modalDescription('Are you sure you want to approve the selected reviews?')
                        ->successNotification(
                            Notification::make()
                                ->success()
                                ->title('Reviews Approved')
                                ->body('Selected reviews have been approved successfully')
                        ),
                    Tables\Actions\BulkAction::make('disapprove')
                        ->label('Disapprove Selected')
                        ->icon('heroicon-o-x-circle')
                        ->color('warning')
                        ->action(function (\Illuminate\Database\Eloquent\Collection $records): void {
                            $records->each(function (Review $record) {
                                $record->status = false;
                                $record->save();
                            });
                        })
                        ->requiresConfirmation()
                        ->modalHeading('Disapprove Selected Reviews')
                        ->modalDescription('Are you sure you want to disapprove the selected reviews?')
                        ->successNotification(
                            Notification::make()
                                ->warning()
                                ->title('Reviews Disapproved')
                                ->body('Selected reviews have been disapproved successfully')
                        ),
                    DeleteBulkAction::make(),
                    ForceDeleteBulkAction::make(),
                    RestoreBulkAction::make(),
                ]),
            ])
            ->paginated([10, 25, 50, 100])
            ->defaultPaginationPageOption(10)
            ->deferLoading()
            ->defaultSort('created_at', 'desc');
    }

    public function getTableQuery(): Builder
    {
        return $this->member->reviews()
            ->with(['user', 'ratings'])
            ->getQuery();
    }

    public function render()
    {
        return view('livewire.member.tabs.reviews-tab');
    }
}
