<?php

namespace App\Livewire\Member\Tabs;

use Livewire\Component;
use App\Models\User\User;
use App\Models\Membership\Subscription;
use App\Enums\SubscriptionStatus;
use App\Enums\MembershipPaymentTerm;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Filament\Tables\Concerns\InteractsWithTable;
use Filament\Tables\Contracts\HasTable;
use Filament\Support\Enums\MaxWidth;
use Filament\Tables\Actions\Action;
use Filament\Notifications\Notification;
use Filament\Support\Contracts\TranslatableContentDriver;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Actions\ViewAction;
use Filament\Tables\Actions\EditAction;
use Filament\Tables\Actions\DeleteAction;
use Filament\Tables\Actions\BulkActionGroup;
use Filament\Tables\Actions\DeleteBulkAction;
use Filament\Tables\Actions\ActionGroup;
use Illuminate\Support\Str;

class SubscriptionsTab extends Component implements HasTable, HasForms
{
    use InteractsWithTable;
    use InteractsWithForms;

    public User $member;

    public function mount(User $member): void
    {
        $this->member = $member;
    }

    public function makeFilamentTranslatableContentDriver(): ?TranslatableContentDriver
    {
        return null;
    }

    public function table(Table $table): Table
    {
        return $table
            ->query($this->getTableQuery())
            ->columns([
                Tables\Columns\TextColumn::make('membershipPlan.package_title')
                    ->label('Plan')
                    ->searchable()
                    ->formatStateUsing(fn($state) => Str::limit($state, 25)),
                Tables\Columns\BadgeColumn::make('payment_term')
                    ->label('Payment Term')
                    ->badge()
                    ->colors([
                        'primary' => MembershipPaymentTerm::Monthly->value,
                        'success' => MembershipPaymentTerm::Yearly->value,
                        'warning' => MembershipPaymentTerm::Quarterly->value,
                        'info' => MembershipPaymentTerm::SemiYearly->value,
                    ]),
                Tables\Columns\BadgeColumn::make('status')
                    ->label('Status')
                    ->badge()
                    ->colors([
                        'success' => SubscriptionStatus::Active->value,
                        'danger' => SubscriptionStatus::Canceled->value,
                    ]),
                Tables\Columns\IconColumn::make('is_active')
                    ->label('Active')
                    ->boolean()
                    ->trueIcon('heroicon-o-check-circle')
                    ->falseIcon('heroicon-o-x-circle')
                    ->trueColor('success')
                    ->falseColor('danger'),
                Tables\Columns\IconColumn::make('auto_renewal')
                    ->label('Auto Renewal')
                    ->boolean()
                    ->trueIcon('heroicon-o-arrow-path')
                    ->falseIcon('heroicon-o-x-mark')
                    ->trueColor('success')
                    ->falseColor('gray'),
                Tables\Columns\TextColumn::make('current_period_start')
                    ->label('Start Date')
                    ->date()
                    ->sortable(),
                Tables\Columns\TextColumn::make('current_period_end')
                    ->label('End Date')
                    ->date()
                    ->sortable(),
                Tables\Columns\TextColumn::make('purchase_date')
                    ->label('Purchase Date')
                    ->date()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('trial_ends_at')
                    ->label('Trial Ends')
                    ->date()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('cancelled_at')
                    ->label('Cancelled Date')
                    ->date()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Tables\Filters\SelectFilter::make('status')
                    ->options([
                        SubscriptionStatus::Active->value => SubscriptionStatus::Active->getLabel(),
                        SubscriptionStatus::Canceled->value => SubscriptionStatus::Canceled->getLabel(),
                    ])
                    ->label('Status'),
                Tables\Filters\SelectFilter::make('payment_term')
                    ->options([
                        MembershipPaymentTerm::Monthly->value => MembershipPaymentTerm::Monthly->getLabel(),
                        MembershipPaymentTerm::Quarterly->value => MembershipPaymentTerm::Quarterly->getLabel(),
                        MembershipPaymentTerm::SemiYearly->value => MembershipPaymentTerm::SemiYearly->getLabel(),
                        MembershipPaymentTerm::Yearly->value => MembershipPaymentTerm::Yearly->getLabel(),
                    ])
                    ->label('Payment Term'),
                Tables\Filters\TernaryFilter::make('is_active')
                    ->label('Active')
                    ->placeholder('All subscriptions')
                    ->trueLabel('Active only')
                    ->falseLabel('Inactive only'),
                Tables\Filters\TernaryFilter::make('auto_renewal')
                    ->label('Auto Renewal')
                    ->placeholder('All subscriptions')
                    ->trueLabel('Auto renewal enabled')
                    ->falseLabel('Auto renewal disabled'),
            ])
            ->filtersFormColumns(2)
            ->filtersFormWidth(MaxWidth::Medium)
            ->actions([
                ActionGroup::make([
                    Action::make('activate')
                        ->icon('heroicon-o-check-circle')
                        ->color('success')
                        ->visible(
                            fn(Subscription $record): bool =>
                            $record->status === SubscriptionStatus::Canceled || !$record->is_active
                        )
                        ->action(function (Subscription $record): void {
                            $record->status = SubscriptionStatus::Active;
                            $record->is_active = true;
                            $record->cancelled_at = null;
                            $record->save();
                        })
                        ->requiresConfirmation()
                        ->modalHeading('Activate Subscription')
                        ->modalDescription('Are you sure you want to activate this subscription?')
                        ->modalSubmitActionLabel('Yes, activate')
                        ->successNotification(
                            Notification::make()
                                ->success()
                                ->title('Subscription Activated')
                                ->body('Subscription has been activated successfully')
                        ),
                    Action::make('cancel')
                        ->icon('heroicon-o-x-circle')
                        ->color('danger')
                        ->visible(
                            fn(Subscription $record): bool =>
                            $record->status === SubscriptionStatus::Active
                        )
                        ->action(function (Subscription $record): void {
                            $record->status = SubscriptionStatus::Canceled;
                            $record->is_active = false;
                            $record->cancelled_at = now();
                            $record->save();
                        })
                        ->requiresConfirmation()
                        ->modalHeading('Cancel Subscription')
                        ->modalDescription('Are you sure you want to cancel this subscription? This action can be reversed.')
                        ->modalSubmitActionLabel('Yes, cancel')
                        ->successNotification(
                            Notification::make()
                                ->warning()
                                ->title('Subscription Cancelled')
                                ->body('Subscription has been cancelled successfully')
                        ),
                    Action::make('toggleAutoRenewal')
                        ->label(fn(Subscription $record): string => $record->auto_renewal ? 'Disable Auto Renewal' : 'Enable Auto Renewal')
                        ->icon(fn(Subscription $record): string => $record->auto_renewal ? 'heroicon-o-x-mark' : 'heroicon-o-arrow-path')
                        ->color(fn(Subscription $record): string => $record->auto_renewal ? 'warning' : 'success')
                        ->action(function (Subscription $record): void {
                            $record->auto_renewal = !$record->auto_renewal;
                            $record->save();
                        })
                        ->requiresConfirmation()
                        ->modalHeading(fn(Subscription $record): string => $record->auto_renewal ? 'Disable Auto Renewal' : 'Enable Auto Renewal')
                        ->modalDescription(fn(Subscription $record): string => $record->auto_renewal
                            ? 'Are you sure you want to disable auto renewal for this subscription?'
                            : 'Are you sure you want to enable auto renewal for this subscription?')
                        ->modalSubmitActionLabel(fn(Subscription $record): string => $record->auto_renewal ? 'Yes, disable' : 'Yes, enable')
                        ->successNotification(
                            Notification::make()
                                ->success()
                                ->title('Auto Renewal Updated')
                                ->body('Auto renewal setting has been updated successfully')
                        ),
                    ViewAction::make()
                        ->url(fn(Subscription $record): string => route('filament.espadmin.resources.membership.subscriptions.view', ['record' => $record->id]))
                        ->openUrlInNewTab(),
                    EditAction::make()
                        ->url(fn(Subscription $record): string => route('filament.espadmin.resources.membership.subscriptions.edit', ['record' => $record->id]))
                        ->openUrlInNewTab(),
                    DeleteAction::make()
                        ->requiresConfirmation()
                        ->modalHeading('Delete Subscription')
                        ->modalDescription('Are you sure you want to delete this subscription? This action cannot be undone.')
                        ->successNotification(
                            Notification::make()
                                ->success()
                                ->title('Subscription Deleted')
                                ->body('Subscription has been deleted successfully')
                        ),
                ])->tooltip('Actions'),
            ])
            ->bulkActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make()
                        ->requiresConfirmation()
                        ->modalHeading('Delete Selected Subscriptions')
                        ->modalDescription('Are you sure you want to delete the selected subscriptions? This action cannot be undone.'),
                ]),
            ])
            ->paginated([10, 25, 50, 100])
            ->defaultPaginationPageOption(10)
            ->deferLoading()
            ->defaultSort('created_at', 'desc');
    }

    public function getTableQuery(): Builder
    {
        return $this->member->subscriptions()->getQuery();
    }

    public function render()
    {
        return view('livewire.member.tabs.subscriptions-tab');
    }
}
