<?php

namespace App\Livewire\Member\Tabs;

use Livewire\Component;
use App\Models\User\User;
use App\Models\Team\TeamMember;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Columns\SpatieMediaLibraryImageColumn;
use Illuminate\Database\Eloquent\Builder;
use Filament\Tables\Concerns\InteractsWithTable;
use Filament\Tables\Contracts\HasTable;
use App\Enums\TeamMemberStatus;
use Filament\Tables\Actions\Action;
use Filament\Tables\Actions\ActionGroup;
use Filament\Tables\Actions\EditAction;
use Filament\Tables\Actions\DeleteAction;
use Filament\Tables\Actions\ForceDeleteAction;
use Filament\Tables\Actions\RestoreAction;
use Filament\Tables\Actions\ReplicateAction;
use Filament\Tables\Actions\BulkActionGroup;
use Filament\Tables\Actions\DeleteBulkAction;
use Filament\Tables\Actions\ForceDeleteBulkAction;
use Filament\Tables\Actions\RestoreBulkAction;
use Filament\Support\Enums\MaxWidth;
use Filament\Notifications\Notification;
use Filament\Support\Contracts\TranslatableContentDriver;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Tables\Actions\ViewAction;

class TeamMembersTab extends Component implements HasTable, HasForms
{
    use InteractsWithTable;
    use InteractsWithForms;

    public User $member;

    public function mount(User $member)
    {
        $this->member = $member;
    }

    public function makeFilamentTranslatableContentDriver(): ?TranslatableContentDriver
    {
        return null;
    }

    public function table(Table $table): Table
    {
        return $table
            ->query($this->getTableQuery())
            ->columns([
                SpatieMediaLibraryImageColumn::make('featured_image')
                    ->collection('featured_image')
                    ->conversion('thumb')
                    ->size(40)
                    ->label('Image')
                    ->square(),
                TextColumn::make('name')
                    ->label('Name')
                    ->getStateUsing(fn (TeamMember $record): string => 
                        trim("{$record->first_name} {$record->last_name}")
                    )
                    ->searchable(['first_name', 'last_name'])
                    ->sortable(['first_name']),
                TextColumn::make('email')
                    ->searchable()
                    ->sortable(),
                TextColumn::make('phone')
                    ->searchable(),
                TextColumn::make('experience_years')
                    ->label('Experience')
                    ->suffix(' Years')
                    ->sortable(),
                Tables\Columns\TextColumn::make('team_member_status')
                    ->badge()
                    ->label('Status'),
                TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                TextColumn::make('updated_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Tables\Filters\TrashedFilter::make(),
                Tables\Filters\SelectFilter::make('team_member_status')
                    ->options([
                        TeamMemberStatus::Draft->value => TeamMemberStatus::Draft->name,
                        TeamMemberStatus::Published->value => TeamMemberStatus::Published->name,
                        TeamMemberStatus::Pending->value => TeamMemberStatus::Pending->name,
                        TeamMemberStatus::Expired->value => TeamMemberStatus::Expired->name,
                    ])
                    ->label('Status'),
            ])
            ->filtersFormColumns(2)
            ->filtersFormWidth(MaxWidth::Medium)

            ->actions([
                ActionGroup::make([
                    
                    Action::make('approve')
                        ->icon('heroicon-o-check-circle')
                        ->color('success')
                        ->visible(fn (TeamMember $record): bool => 
                            $record->team_member_status->value === TeamMemberStatus::Pending->value
                        )
                        ->action(function (TeamMember $record): void {
                            $record->team_member_status = TeamMemberStatus::Published->value;
                            $record->published_at = now();
                            $record->save();
                        })
                        ->requiresConfirmation()
                        ->modalHeading('Approve Team Member')
                        ->modalDescription('Are you sure you want to approve this team member? This will make it publicly visible.')
                        ->modalSubmitActionLabel('Yes, approve')
                        ->successNotification(
                            Notification::make()
                                ->success()
                                ->title('Team Member Approved')
                                ->body('Team Member has been approved successfully')
                        ),
                    Action::make('publish')
                        ->icon('heroicon-o-arrow-up-circle')
                        ->color('success')
                        ->visible(fn (TeamMember $record): bool => 
                            in_array($record->team_member_status->value, [TeamMemberStatus::Draft->value, TeamMemberStatus::Expired->value])
                        )
                        ->action(function (TeamMember $record): void {
                            $record->team_member_status = TeamMemberStatus::Published->value;
                            $record->published_at = now();
                            $record->save();
                        })
                        ->requiresConfirmation()
                        ->modalHeading('Publish Team Member')
                        ->modalDescription('Are you sure you want to publish this team member? This will make it publicly visible.')
                        ->modalSubmitActionLabel('Yes, Publish')
                        ->successNotification(
                            Notification::make()
                                ->success()
                                ->title('Team Member Published')
                                ->body('Team Member has been published successfully')
                        ),
                    EditAction::make() 
                        ->url(fn (TeamMember $record): string => route('filament.espadmin.resources.team.team-members.edit', ['record' => $record->id]))
                        ->openUrlInNewTab()
                        ->hidden(fn(TeamMember $record): bool => $record->trashed()),
                    ReplicateAction::make()
                        ->before(function (TeamMember $record): void {
                            $record->team_member_status = TeamMemberStatus::Draft->value;
                        })
                        ->hidden(fn(TeamMember $record): bool => $record->trashed()),
                    DeleteAction::make()
                        ->hidden(fn(TeamMember $record): bool => $record->trashed()),
                    ForceDeleteAction::make(),
                    RestoreAction::make(),
                ])->tooltip('Actions'),
            ])
            ->bulkActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                    ForceDeleteBulkAction::make(),
                    RestoreBulkAction::make(),
                ]),
            ])
            ->paginated([10, 25, 50, 100])
            ->defaultPaginationPageOption(10)
            ->deferLoading()
            ->defaultSort('created_at', 'desc');
    }

    public function getTableQuery(): Builder
    {
        return $this->member->team()->getQuery();
    }

    public function render()
    {
        return view('livewire.member.tabs.team-members-tab');
    }
} 