<?php

namespace App\Livewire\Member\Tabs;

use Livewire\Component;
use App\Models\User\User;
use App\Models\Membership\Purchase;
use App\Enums\PurchaseItemType;
use App\Enums\PaymentStatus;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Select;
use Illuminate\Database\Eloquent\Builder;
use Filament\Tables\Concerns\InteractsWithTable;
use Filament\Tables\Contracts\HasTable;
use Filament\Support\Enums\MaxWidth;
use Filament\Tables\Actions\Action;
use Filament\Notifications\Notification;
use Filament\Support\Contracts\TranslatableContentDriver;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Actions\ViewAction;
use Filament\Tables\Actions\ActionGroup;
use Illuminate\Support\Str;

class TransactionsTab extends Component implements HasTable, HasForms
{
    use InteractsWithTable;
    use InteractsWithForms;

    public User $member;

    public function mount(User $member): void
    {
        $this->member = $member;
    }

    public function makeFilamentTranslatableContentDriver(): ?TranslatableContentDriver
    {
        return null;
    }

    public function table(Table $table): Table
    {
        return $table
            ->query($this->getTableQuery())
            ->columns([
                Tables\Columns\TextColumn::make('purchase_date')
                    ->label('Date')
                    ->dateTime()
                    ->sortable()
                    ->since()
                    ->toggleable(),
                Tables\Columns\TextColumn::make('item_type')
                    ->label('Type')
                    ->badge()
                    ->searchable(),
                Tables\Columns\TextColumn::make('purchaseable_details')
                    ->label('Item')
                    ->getStateUsing(function (Purchase $record) {
                        if ($record->purchaseable_type === 'App\\Models\\Membership\\Subscription') {
                            $subscription = $record->purchaseable;
                            return $subscription?->membershipPlan?->package_title ?? 'Unknown Plan';
                        }
                        return $record->purchaseable_type ? class_basename($record->purchaseable_type) : 'N/A';
                    })
                    ->wrap()
                    ->searchable(),
                Tables\Columns\TextColumn::make('amount')
                    ->label('Amount')
                    ->money('USD')
                    ->sortable(),
                Tables\Columns\TextColumn::make('total_amount')
                    ->label('Total')
                    ->money('USD')
                    ->sortable(),
                Tables\Columns\TextColumn::make('payment_status')
                    ->label('Payment Status')
                    ->getStateUsing(function (Purchase $record) {
                        $latestPayment = $record->latestPayment();
                        return $latestPayment?->status?->getLabel() ?? 'No Payment';
                    })
                    ->badge()
                    ->color(fn(string $state): string => match ($state) {
                        'Paid' => 'success',
                        'Pending' => 'warning',
                        'Failed' => 'danger',
                        'Refunded' => 'info',
                        default => 'gray',
                    }),
                Tables\Columns\IconColumn::make('status')
                    ->label('Purchase Status')
                    ->boolean()
                    ->trueIcon('heroicon-o-check-circle')
                    ->falseIcon('heroicon-o-x-circle')
                    ->trueColor('success')
                    ->falseColor('danger'),
                Tables\Columns\TextColumn::make('payments_count')
                    ->label('Payments')
                    ->counts('payments')
                    ->badge()
                    ->color('primary')
                    ->toggleable(),
                Tables\Columns\TextColumn::make('notes')
                    ->label('Notes')
                    ->limit(30)
                    ->tooltip(function (TextColumn $column): ?string {
                        $state = $column->getState();
                        if (strlen($state) <= 30) {
                            return null;
                        }
                        return $state;
                    })
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Tables\Filters\SelectFilter::make('item_type')
                    ->label('Type')
                    ->options([
                        PurchaseItemType::Subscription->value => PurchaseItemType::Subscription->getLabel(),
                        PurchaseItemType::Credit->value => PurchaseItemType::Credit->getLabel(),
                        PurchaseItemType::Listing->value => PurchaseItemType::Listing->getLabel(),
                    ]),
                Tables\Filters\TernaryFilter::make('status')
                    ->label('Purchase Status')
                    ->placeholder('All purchases')
                    ->trueLabel('Successful only')
                    ->falseLabel('Failed only'),
                Tables\Filters\Filter::make('payment_status')
                    ->form([
                        Select::make('payment_status')
                            ->label('Payment Status')
                            ->options([
                                PaymentStatus::Paid->value => PaymentStatus::Paid->getLabel(),
                                PaymentStatus::Pending->value => PaymentStatus::Pending->getLabel(),
                                PaymentStatus::Failed->value => PaymentStatus::Failed->getLabel(),
                                PaymentStatus::Refunded->value => PaymentStatus::Refunded->getLabel(),
                            ])
                    ])
                    ->query(function (Builder $query, array $data): Builder {
                        return $query->when(
                            $data['payment_status'],
                            fn(Builder $query, $status): Builder => $query->whereHas(
                                'payments',
                                fn(Builder $query): Builder => $query->where('status', $status)
                            )
                        );
                    }),
                Tables\Filters\Filter::make('amount_range')
                    ->form([
                        TextInput::make('amount_from')
                            ->label('Amount From')
                            ->numeric(),
                        TextInput::make('amount_to')
                            ->label('Amount To')
                            ->numeric(),
                    ])
                    ->query(function (Builder $query, array $data): Builder {
                        return $query
                            ->when(
                                $data['amount_from'],
                                fn(Builder $query, $amount): Builder => $query->where('total_amount', '>=', $amount)
                            )
                            ->when(
                                $data['amount_to'],
                                fn(Builder $query, $amount): Builder => $query->where('total_amount', '<=', $amount)
                            );
                    }),
            ])
            ->filtersFormColumns(2)
            ->filtersFormWidth(MaxWidth::Medium)
            ->actions([
                ActionGroup::make([
                    ViewAction::make()
                        ->label('View Details')
                        ->modalHeading('Purchase Details')
                        ->modalContent(function (Purchase $record) {
                            return view('filament.resources.member.modals.purchase-details', compact('record'));
                        })
                        ->modalWidth(MaxWidth::Large),
                    Action::make('view_payments')
                        ->label('View Payments')
                        ->icon('heroicon-o-credit-card')
                        ->color('primary')
                        ->modalHeading('Payment History')
                        ->modalContent(function (Purchase $record) {
                            return view('filament.resources.member.modals.payment-history', compact('record'));
                        })
                        ->modalWidth(MaxWidth::Large),
                    Action::make('view_item')
                        ->label('View Item')
                        ->icon('heroicon-o-eye')
                        ->color('info')
                        ->visible(fn(Purchase $record): bool => $record->purchaseable !== null)
                        ->url(function (Purchase $record): ?string {
                            if ($record->purchaseable_type === 'App\\Models\\Membership\\Subscription') {
                                return route('filament.espadmin.resources.membership.subscriptions.view', [
                                    'record' => $record->purchaseable_id
                                ]);
                            }
                            return null;
                        })
                        ->openUrlInNewTab(),
                ])->tooltip('Actions'),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\BulkAction::make('export')
                        ->label('Export Selected')
                        ->icon('heroicon-o-document-arrow-down')
                        ->action(function (\Illuminate\Database\Eloquent\Collection $records): void {
                            // This would implement export functionality
                            Notification::make()
                                ->success()
                                ->title('Export Started')
                                ->body('Export functionality would be implemented here')
                                ->send();
                        }),
                ]),
            ])
            ->paginated([10, 25, 50, 100])
            ->defaultPaginationPageOption(10)
            ->deferLoading()
            ->defaultSort('purchase_date', 'desc');
    }

    public function getTableQuery(): Builder
    {
        return $this->member->purchases()
            ->with(['purchaseable', 'payments'])
            ->getQuery();
    }

    public function render()
    {
        return view('livewire.member.tabs.transactions-tab');
    }
}
