<?php

namespace App\Models\Blog;

use App\Auditing\Modifiers\CreatedByModifier;
use App\Auditing\Modifiers\IsFeaturedModifier;
use App\Auditing\Modifiers\PostDescriptionModifier;
use App\Enums\PostStatus;
use App\Models\Blog\PostCategory;
use App\Models\Blog\PostTag;
use App\Models\Blog\Traits\PostHelperTrait;
use App\Models\Traits\CreatedByTrait;
use App\Models\Traits\HasUuidTrait;
use App\Models\User\User;
use App\Observers\PostObserver;
use Biostate\FilamentMenuBuilder\Traits\Menuable;
use Illuminate\Database\Eloquent\Attributes\ObservedBy;
use Illuminate\Database\Eloquent\Attributes\Scope;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\MorphToMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use OwenIt\Auditing\Auditable as AuditableTrait;
use OwenIt\Auditing\Contracts\Auditable;
use RalphJSmit\Laravel\SEO\SchemaCollection;
use RalphJSmit\Laravel\SEO\Support\HasSEO;
use RalphJSmit\Laravel\SEO\Support\SEOData;
use Spatie\Image\Enums\Fit;
use Spatie\MediaLibrary\HasMedia;
use Spatie\MediaLibrary\InteractsWithMedia;
use Spatie\MediaLibrary\MediaCollections\Models\Media;
use Spatie\Sluggable\HasSlug;
use Spatie\Sluggable\SlugOptions;

#[ObservedBy(PostObserver::class)]
class Post extends Model implements HasMedia, Auditable
{
    use HasFactory, SoftDeletes, InteractsWithMedia, HasSlug, HasUuidTrait, CreatedByTrait, HasSEO, Menuable, AuditableTrait, PostHelperTrait;

    protected $fillable = [
        'post_title',
        'post_excerpt',
        'post_description',
        'slug',
        'post_status',
        'is_featured',
        'created_by',
        'published_at',
    ];

    protected $casts = [
        'is_featured'      => 'boolean',
        'post_description' => 'array',
        'post_status'      => PostStatus::class,
        'published_at'     => 'datetime',
        'expired_at'       => 'datetime',
    ];

    protected $auditInclude = [
        'post_title',
        'post_excerpt',
        'post_description',
        'slug',
        'post_status',
        'is_featured',
        'created_by',
        'published_at',
    ];

    /**
     * Attribute modifiers.
     *
     * @var array
     */
    protected $attributeModifiers = [
        'is_featured'      => IsFeaturedModifier::class,
        'created_by'       => CreatedByModifier::class,
        'post_description' => PostDescriptionModifier::class,
    ];

    public function getSlugOptions(): SlugOptions
    {
        return SlugOptions::create()->generateSlugsFrom('post_title')->saveSlugsTo('slug')->doNotGenerateSlugsOnUpdate();
    }

    public function tags(): BelongsToMany
    {
        return $this->belongsToMany(PostTag::class, 'post_post_tag');
    }

    public function categories(): BelongsToMany
    {
        return $this->belongsToMany(PostCategory::class, 'post_post_category');
    }

    public function author(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function favoritesUsers(): MorphToMany
    {
        return $this->morphToMany(User::class, 'model');
    }

    public function getPermalinkAttribute(): string
    {
        if ($this->slug) {
            return route('journal.show', ['slug' => $this->slug]);
        }
        return 'abc';
    }



    public function getPostDescriptionTextAttribute(): string
    {
        return tiptap_converter()->asText($this->post_description);
    }

    public function getPostDescriptionHtmlAttribute(): string
    {
        return tiptap_converter()->asHTML($this->post_description);
    }
    //
    #[Scope]
    public function featured(Builder $query): void
    {
        $query->where('is_featured', 1);
    }
    #[Scope]
    public function latest(Builder $query): void
    {
        $query->whereNotNull('published_at')->where('published_at', '>=', now()->subDays(24));
    }

    #[Scope]
    public function popular(Builder $query): void
    {
        $query->whereNotNull('updated_at');
    }


    public function getDynamicSEOData(): SEOData
    {
        $pathToFeaturedImageRelativeToPublicPath = $this->getFirstMediaUrl('featured_image');

        $seoTitle       = empty($this->seo->title) ? $this->post_title : $this->seo->title;
        $seoDescription = empty($this->seo->description) ? $this->post_excerpt : $this->seo->description;

        // Override only the properties you want:
        return new SEOData(
            title: $seoTitle,
            description: $seoDescription,
            image: $pathToFeaturedImageRelativeToPublicPath,
            author: $this->author->name,
            schema: SchemaCollection::make()->addArticle(),
        );
    }

    // Define media collections
    public function registerMediaCollections(): void
    {
        $this->addMediaCollection('featured_image')
            ->singleFile()
            ->useFallbackUrl('/images/image_placeholder.webp')
            ->useFallbackPath(public_path('/images/image_placeholder.webp'))
            ->registerMediaConversions(function (Media $media) {
                $this->addMediaConversion('thumb')
                    ->fit(Fit::Contain, 400)
                    ->sharpen(10);
                $this->addMediaConversion('medium')
                    ->width(768)
                    ->sharpen(10);
                $this->addMediaConversion('large')
                    ->width(1024)
                    ->height(768)
                    ->sharpen(10);
                $this->addMediaConversion('cover_image')
                    ->width(1280)
                    ->sharpen(20);
            });
    }

    public function scopeScheduledForPublishing($query)
    {
        return $query
            ->where('post_status', PostStatus::Scheduled->value)
            ->where('published_at', '<=', now());
    }

    public function getMenuLinkAttribute(): string
    {
        return route('journal.show', $this);
    }

    public function getMenuNameAttribute(): string
    {
        return $this->post_title;
    }

    public function scopePublished($query)
    {
        return $query->where('post_status', PostStatus::Published->value);
    }

    public function scopeFeatured($query)
    {
        return $query->where('is_featured', true);
    }

    public function isPublished(): bool
    {
        return $this->post_status === PostStatus::Published;
    }

    public function canEdit(?User $user = null): bool
    {
        if (! $user) {
            return false;
        }

        // User can edit if they are the author or have the edit_posts permission
        return $user->id === $this->created_by || $user->can('edit_posts');
    }
}
