<?php

namespace App\Models\Blog;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Spatie\Sluggable\HasSlug;
use Spatie\Sluggable\SlugOptions;
use App\Models\Traits\HasUuidTrait;
use App\Models\Traits\CreatedByTrait;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use RalphJSmit\Laravel\SEO\Support\HasSEO;
use RalphJSmit\Laravel\SEO\Support\SEOData;
use App\Observers\PostCategoryObserver;
use Illuminate\Database\Eloquent\Attributes\ObservedBy;
use Illuminate\Support\Str;

#[ObservedBy(PostCategoryObserver::class)]
class PostCategory extends Model
{
    use HasFactory, SoftDeletes, HasSlug, HasUuidTrait, CreatedByTrait, HasSEO;

    protected $fillable = [
        'name',
        'slug',
        'description',
        'meta_title',
        'meta_description',
        'is_active',
        'color',
        'created_by',
        'uuid',
        'parent_id'
    ];

    public function getSlugOptions(): SlugOptions
    {
        return SlugOptions::create()->generateSlugsFrom('name')->saveSlugsTo('slug')->doNotGenerateSlugsOnUpdate();
    }

    public function posts(): BelongsToMany
    {
        return $this->belongsToMany(Post::class, 'post_post_category');
    }

    public function parent(): BelongsTo
    {
        return $this->belongsTo(PostCategory::class, 'parent_id');
    }

    public function children(): HasMany
    {
        return $this->hasMany(PostCategory::class, 'parent_id');
    }

    public function getPermalinkAttribute(): string
    {
        return route('journal.category', ['slug' =>  $this->slug]);
    }

    public function getDynamicSEOData(): SEOData
    {
        $image = asset('images/esp_logo.png');
        $category_name = $this->name;
        $category_description = Str::limit(strip_tags($this->description), 155);
        $meta_title = $this->seo->title ?? $category_name . " - ESP Marketplace Journal";
        $meta_description = $this->seo->description ?? $category_description;

        // Override only the properties you want:
        return new SEOData(
            title: $meta_title,
            description: $meta_description,
            image: $image,
        );
    }
}
