<?php

namespace App\Models\Blog;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Spatie\Sluggable\HasSlug;
use Spatie\Sluggable\SlugOptions;
use App\Models\Traits\HasUuidTrait;
use App\Models\Traits\CreatedByTrait;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use RalphJSmit\Laravel\SEO\Support\HasSEO;
use RalphJSmit\Laravel\SEO\Support\SEOData;
use Illuminate\Support\Str;

class PostTag extends Model
{
    use HasFactory, SoftDeletes, HasSlug, HasUuidTrait, CreatedByTrait, HasSEO;

    protected $fillable = [
        'name',
        'slug',
        'description',
        'created_by'
    ];

    public function getSlugOptions(): SlugOptions
    {
        return SlugOptions::create()
            ->generateSlugsFrom('name')
            ->saveSlugsTo('slug')
            ->doNotGenerateSlugsOnUpdate();
    }

    public function posts(): BelongsToMany
    {
        return $this->belongsToMany(Post::class, 'post_post_tag');
    }

    public function getPermalinkAttribute(): string
    {
        return route('journal.tag', ['slug' => $this->slug]);
    }

    public function getDynamicSEOData(): SEOData
    {
        $image = asset('images/esp_logo.png');
        $tag_name = $this->name;
        $tag_description = Str::limit(strip_tags($this->description), 155);

        // Override only the properties you want:
        return new SEOData(
            title: $tag_name . " - ESP Marketplace Journal",
            description: $tag_description,
            image: $image,
        );
    }
}
