<?php

namespace App\Models\Classified;

use App\Enums\ClassifiedCondition;
use App\Enums\ClassifiedLabel;
use App\Enums\ClassifiedStatus;
use App\Models\Classified\Traits\ClassifiedHelperTrait;
use App\Models\Location\City;
use App\Models\Location\Country;
use App\Models\Location\State;
use App\Models\Traits\CreatedByTrait;
use App\Models\Traits\HasUuidTrait;
use App\Models\User\User;
use App\Observers\ClassifiedObserver;
use Illuminate\Database\Eloquent\Attributes\ObservedBy;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\MorphToMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use OwenIt\Auditing\Auditable as AuditableTrait;
use OwenIt\Auditing\Contracts\Auditable;
use RalphJSmit\Laravel\SEO\Support\HasSEO;
use Spatie\Image\Enums\Fit;
use Spatie\MediaLibrary\HasMedia;
use Spatie\MediaLibrary\InteractsWithMedia;
use Spatie\MediaLibrary\MediaCollections\Models\Media;
use Spatie\Sluggable\HasSlug;
use Spatie\Sluggable\SlugOptions;

#[ObservedBy(ClassifiedObserver::class)]
class Classified extends Model implements HasMedia, Auditable
{
    use HasSlug, HasUuidTrait, CreatedByTrait, SoftDeletes, InteractsWithMedia, HasFactory, HasSEO, AuditableTrait, ClassifiedHelperTrait;

    protected $fillable = [
        'uuid',
        'user_id',
        'title',
        'slug',
        'description',
        'price',
        'discount_price',
        'key_features',
        'maintenance_detail',
        'website_link',
        'is_featured',
        'discount_start',
        'discount_end',
        'location',
        'address',
        'city_id',
        'state_id',
        'country_id',
        'postal_code',
        'latitude',
        'longitude',
        'meta_title',
        'meta_description',
        'created_by',
        'published_at',
        'classified_status',
        'condition',
        'map_address',
        'manufacture_date',
        'label',

    ];

    protected $casts = [
        'gallery'           => 'array',
        'maintenance_docs'  => 'array',
        'key_features'      => 'array',
        'is_featured'       => 'boolean',
        'classified_status' => ClassifiedStatus::class,
        'condition'         => ClassifiedCondition::class,
        'label'             => ClassifiedLabel::class,
        'discount_start'    => 'datetime',
        'discount_end'      => 'datetime',
    ];

    protected $auditInclude = [
        'title',
        'slug',
        'classified_status',
        'published_at',
    ];

    protected $appends = [
        'map_address',
    ];
    /**
     * Scopes
     *
     * @return void
     */

    public function scopeOnlyFlashSale(Builder $query): void
    {
        $now = now();
        $query->where('discount_start', '<=', $now)
            ->where('discount_end', '>=', $now)
            ->where('discount_price', '>', 0);
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class,'user_id');
    }

    public function getSlugOptions(): SlugOptions
    {
        return SlugOptions::create()->generateSlugsFrom('title')->saveSlugsTo('slug')->doNotGenerateSlugsOnUpdate();
    }

    public function author(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function tags(): BelongsToMany
    {
        return $this->belongsToMany(ClassifiedTag::class, 'classified_classified_tag');
    }

    public function types(): BelongsToMany
    {
        return $this->belongsToMany(ClassifiedType::class, 'classified_classified_type');
    }

    public function categories(): BelongsToMany
    {
        return $this->belongsToMany(ClassifiedCategory::class, 'classified_classified_category');
    }

    public function city(): BelongsTo
    {
        return $this->belongsTo(City::class);
    }

    public function state(): BelongsTo
    {
        return $this->belongsTo(State::class);
    }

    public function country(): BelongsTo
    {
        return $this->belongsTo(Country::class);
    }

    public function deliveries(): BelongsToMany
    {
        return $this->belongsToMany(ClassifiedDelivery::class);
    }

    public function favoritesUsers(): MorphToMany
    {
        return $this->morphToMany(User::class, 'model');
    }

    public function registerMediaCollections(): void
    {
        $this->addMediaCollection('featured_image')
            ->singleFile()
            ->useFallbackUrl('/images/image_placeholder.webp')
            ->useFallbackPath(public_path('/images/image_placeholder.webp'))
            ->registerMediaConversions(function (Media $media) {
                $this->addMediaConversion('thumb')
                    ->fit(Fit::Contain, 300)
                    ->keepOriginalImageFormat()
                    ->sharpen(10);
                $this->addMediaConversion('medium')
                    ->fit(Fit::Contain, 768)
                    ->keepOriginalImageFormat()
                    ->sharpen(10);
                $this->addMediaConversion('large')
                    ->fit(Fit::Contain, 1024)
                    ->keepOriginalImageFormat()
                    ->sharpen(10);
            });

        $this->addMediaCollection('gallery')
            ->registerMediaConversions(function (Media $media) {
                $this->addMediaConversion('thumb')
                    ->fit(Fit::Contain, 300)
                    ->keepOriginalImageFormat()
                    ->sharpen(10);
                $this->addMediaConversion('medium')
                    ->fit(Fit::Contain, 600)
                    ->keepOriginalImageFormat()
                    ->sharpen(10);
                $this->addMediaConversion('large')
                    ->fit(Fit::Contain, 1920)
                    ->keepOriginalImageFormat()
                    ->sharpen(10);
            });

        $this->addMediaCollection('maintenance_docs');
        $this->addMediaCollection('video')
            ->acceptsMimeTypes([
                'video/mp4',
                'video/webm',
                'video/ogg',
                'video/avi',
                'video/mov',
                'video/wmv',
                'video/flv',
            ])
            ->useFallbackUrl(asset('/images/image_placeholder.webp'))
            ->useFallbackPath(asset('/images/image_placeholder.webp'))
            ->singleFile()
            ->registerMediaConversions(function (Media $media) {
                $this->addMediaConversion('thumb')
                    ->format('jpg')
                    ->fit(Fit::Contain, 300, null, backgroundColor: '#C2CBD2')
                    ->extractVideoFrameAtSecond(2);
                $this->addMediaConversion('large')
                    ->format('jpg') 
                    ->fit(Fit::Contain, 768, backgroundColor: '#C2CBD2')
                    ->extractVideoFrameAtSecond(2);
            });
    }

    /**
     * Returns the 'latitude' and 'longitude' attributes as the computed 'map_address' attribute,
     * as a standard Google Maps style Point array with 'lat' and 'lng' attributes.
     *
     * Used by the Filament Google Maps package.
     *
     * Requires the 'map_address' attribute be included in this model's $fillable array.
     *
     * @return array
     */

    public function getMapAddressAttribute(): array
    {
        return [
            "lat" => (float) $this->latitude,
            "lng" => (float) $this->longitude,
        ];
    }

    /**
     * Takes a Google style Point array of 'lat' and 'lng' values and assigns them to the
     * 'latitude' and 'longitude' attributes on this model.
     *
     * Used by the Filament Google Maps package.
     *
     * Requires the 'map_address' attribute be included in this model's $fillable array.
     *
     * @param ?array $location
     * @return void
     */
    public function setMapAddressAttribute(?array $location): void
    {
        if (is_array($location)) {
            $this->attributes['latitude']  = $location['lat'];
            $this->attributes['longitude'] = $location['lng'];
            unset($this->attributes['map_address']);
        }
    }

    /**
     * Get the lat and lng attribute/field names used on this table
     *
     * Used by the Filament Google Maps package.
     *
     * @return string[]
     */
    public static function getLatLngAttributes(): array
    {
        return [
            'lat' => 'latitude',
            'lng' => 'longitude',
        ];
    }

    /**
     * Get the name of the computed location attribute
     *
     * Used by the Filament Google Maps package.
     *
     * @return string
     */
    public static function getComputedLocation(): string
    {
        return 'map_address';
    }

    public function isPublished(): bool
    {
        return $this->status === 'published'; // adjust according to your status field
    }
}
