<?php

namespace App\Models\Classified\Traits;

use Carbon\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;

trait ClassifiedHelperTrait
{
    public function getPermalinkAttribute(): string
    {
        return route('classified.show', $this->slug);
    }

    public function getClassifiedCondition()
    {
        $condition = [];
        if (! empty($this->condition)) {
            $condition = esp_classified_get_condition($this, true);
        }
        return $condition;
    }
    public function getClassifiedLabel(): string
    {
        $label = '';
        if (! empty($this->label)) {
            $label = esp_classified_get_label($this, true);
        }
        return $label;
    }

    public function getAddress(string $format = 'short'): string
    {
        $address      = $this->address;
        $addressParts = [];
        $stateCode    = null;
        $countryCode  = null;

        if ($this->address) {
            $addressParts['address_line_1'] = $this->address;
        }

        if ($this->city()->exists()) {
            $addressParts['city'] = $this->city->city_name;
        }

        if ($this->state()->exists()) {
            $addressParts['state'] = $this->state->state_name;
            $stateCode             = $this->state->state_code;
        }

        if ($this->country()->exists()) {
            $addressParts['country'] = $this->country->country_name;
            $countryCode             = $this->country->country_code;
        }
        if ($this->postal_code) {
            $addressParts['postal_code'] = $this->postal_code;
        }

        if ($format == 'short') {
            unset($addressParts['address_line_1']);
            unset($addressParts['postal_code']);
        }

        if ($format == 'area') {
            unset($addressParts['address_line_1']);
            unset($addressParts['country']);
            unset($addressParts['postal_code']);

            if ($stateCode) {
                unset($addressParts['state']);
                $addressParts['state'] = $stateCode;
            }
            if ($countryCode) {
                unset($addressParts['country']);
                $addressParts['country'] = $countryCode;
            }
        }
        return ! empty($addressParts) ? implode(', ', $addressParts) : '';
    }

    public function getCategories($limit = 3): Collection
    {
        $categories = [];
        if (! empty($this->categories)) {
            $categories = $this->categories->take($limit);
        }

        return $categories;
    }

    public function getPostedDate($format = null): string
    {
        $postedDate = Carbon::parse($this->created_at);
        if (! empty($format)) {
            $postedDate = $postedDate->format($format);
        } else {
            $postedDate = $postedDate->diffForHumans();
        }
        return $postedDate;
    }

    public function isUpdated(): bool
    {
        $updatedDate = false;

        if (! empty($this->updated_at)) {
            $updatedDate = true;
        }

        return $updatedDate;
    }

    public function getUpdatedDate($format = null): string
    {
        $updatedDate = Carbon::parse($this->updated_at);
        if (! empty($format)) {
            $updatedDate = $updatedDate->format($format);
        } else {
            $updatedDate = $updatedDate->diffForHumans();
        }
        return $updatedDate;
    }
    public function isNew(int $days = 1, string $column = 'published_at'): bool
    {
        if (empty($this->{$column})) {
            return false;
        }

        $date = Carbon::parse($this->{$column});

        return $date->diffInDays(now()) < $days && $date->lte(now());
    }

    public function getClassifiedTags($limit = null): Collection
    {
        $tags = [];
        if (! empty($limit)) {
            $tags = $this->tags->take($limit);
        } else {
            $tags = $this->tags;
        }

        return $tags;
    }

    public function getClassifiedType($limit = 3): Collection
    {
        $types = [];
        if (! empty($this->types)) {
            $types = $this->types->take($limit);
        }
        return $types;
    }

    public function isFlashSale(): bool
    {
        $isFlashSale = false;
        // must have both endpoints and a positive discount price
        if (
            empty($this->discount_start)
            || empty($this->discount_end)
            || empty($this->discount_price)
            || $this->discount_price <= 0
        ) {
            $isFlashSale = false;
        }

        $start = Carbon::parse($this->discount_start);
        $end   = Carbon::parse($this->discount_end);
        $now   = Carbon::now();
        // 1) current datetime must be between start and end (full datetime precision)
        if ($now->gte($start) && $now->lte($end)) {
            $isFlashSale = true;
        }

        // 2) window must not exceed 48 hours (changed from >= to > to allow exactly 48 hours)
        // if ($start->diffInHours($end) > 48) {
        //     return false;
        // }
        return $isFlashSale;
    }

    public function getKeyFeatures(int $limit = 3): array
    {
        return $this->key_features ?? [];
    }

    public function isFavorite(): bool
    {
        if (! Auth::check()) {
            return false;
        }
        return Auth::user()->favoriteClassified()->where('model_id', $this->id)->exists();
    }
}
