<?php

namespace App\Models\Equipment;

use App\Models\User\User;
use App\Enums\EquipmentStatus;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use App\Models\Location\City;
use App\Models\Location\State;
use App\Models\Location\Country;
use Illuminate\Database\Eloquent\SoftDeletes;
use Spatie\Sluggable\HasSlug;
use Spatie\Sluggable\SlugOptions;
use App\Models\Traits\CreatedByTrait;
use App\Models\Traits\HasUuidTrait;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Spatie\MediaLibrary\HasMedia;
use Spatie\MediaLibrary\InteractsWithMedia;
use Spatie\MediaLibrary\MediaCollections\Models\Media;
use Spatie\Image\Enums\Fit;
use Illuminate\Support\Str;
use RalphJSmit\Laravel\SEO\Support\HasSEO;
use App\Models\Equipment\Traits\EquipmentHelperTrait;
use Illuminate\Database\Eloquent\Relations\MorphToMany;

class Equipment extends Model implements HasMedia
{
    use HasFactory, SoftDeletes, HasSlug, HasUuidTrait, CreatedByTrait, InteractsWithMedia, HasSEO, EquipmentHelperTrait;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'equipments';

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'uuid',
        'user_id',
        'title',
        'slug',
        'address',
        'city_id',
        'state_id',
        'country_id',
        'specification',
        'description',
        'use_case',
        'constraint',
        'tags',
        'status',
        'created_by',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'tags' => 'array',
        'status' => EquipmentStatus::class,
        'specification' => 'array',
    ];

    /**
     * Get the user that owns the equipment.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the categories that the equipment belongs to.
     */
    public function categories(): BelongsToMany
    {
        return $this->belongsToMany(
            EquipmentCategory::class,
            'equipment_equipment_category',
            'equipment_id',
            'equipment_category_id'
        );
    }

    /**
     * Get the city that the equipment belongs to.
     */
    public function city(): BelongsTo
    {
        return $this->belongsTo(City::class);
    }

    /**
     * Get the state that the equipment belongs to.
     */
    public function state(): BelongsTo
    {
        return $this->belongsTo(State::class);
    }

    /**
     * Get the country that the equipment belongs to.
     */
    public function country(): BelongsTo
    {
        return $this->belongsTo(Country::class);
    }

    public function favoritesUsers(): MorphToMany
    {
        return $this->morphToMany(User::class, 'model');
    }

    /**
     * Get the creator of the equipment.
     */
    public function author(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function getSlugOptions(): SlugOptions
    {
        return SlugOptions::create()
            ->generateSlugsFrom('title')
            ->saveSlugsTo('slug')
            ->doNotGenerateSlugsOnUpdate();
    }

    /**
     * Register media collections for the model.
     */
    public function registerMediaCollections(): void
    {
        $this->addMediaCollection('featured_image')
            ->singleFile()
            ->useFallbackUrl(asset('images/image_placeholder.webp'))
            ->useFallbackPath(asset('images/image_placeholder.webp'))
            ->registerMediaConversions(function (Media $media) {
                $this->addMediaConversion('thumb')
                    ->fit(Fit::Contain, 368)
                    ->keepOriginalImageFormat()
                    ->sharpen(10);
                $this->addMediaConversion('medium')
                    ->fit(Fit::Contain, 768)
                    ->keepOriginalImageFormat()
                    ->sharpen(10);
                $this->addMediaConversion('large')
                    ->fit(Fit::Contain, 1024)
                    ->keepOriginalImageFormat()
                    ->sharpen(10);
            });

        $this->addMediaCollection('gallery')
            ->registerMediaConversions(function (Media $media) {
                $this->addMediaConversion('thumb')
                    ->fit(Fit::Contain, 318)
                    ->keepOriginalImageFormat()
                    ->sharpen(10);
                $this->addMediaConversion('medium')
                    ->fit(Fit::Contain, 600)
                    ->keepOriginalImageFormat()
                    ->sharpen(10);
                $this->addMediaConversion('large')
                    ->fit(Fit::Contain, 1920)
                    ->keepOriginalImageFormat()
                    ->sharpen(10);
            });
        $this->addMediaCollection('additional_resources');
    }

    /**
     * Get the first category record associated with the equipment.
     */
    public function getCategoriesLimitAttribute($limit = null)
    {
        // dd($limit);
        if (!empty($limit)) {
            dd($limit);
            $categories = $this->categories()->take($limit)->get();
        } else {
            $categories = $this->categories()->first();
        }

        if ($categories != null) {
            $categories = $categories;
        }

        return $categories;
    }

    /**
     * Get the formatted address attribute.
     */
    public function getFullAddressAttribute()
    {
        $address = '';
        if (!empty($this->country)) {
            $address = $this->country->country_name;
        }
        if (!empty($this->state)) {
            $address = (!empty($address) ? $address . ', ' : '') . $this->state->state_name;
        }
        if (!empty($this->city)) {
            $address = (!empty($address) ? $address . ', ' : '') . $this->city->city_name;
        }
        if (!empty($address)) {
            $address = $address;
        }
        return $address;
    }
}
