<?php

namespace App\Models\Equipment;

use App\Models\User\User;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use App\Models\Traits\HasUuidTrait;
use App\Models\Traits\CreatedByTrait;
use Spatie\Sluggable\HasSlug;
use Spatie\Sluggable\SlugOptions;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;

class EquipmentCategory extends Model
{
    use HasFactory, HasUuidTrait, CreatedByTrait, HasSlug;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'equipment_categories';

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'uuid',
        'parent_id',
        'category_name',
        'slug',
        'category_description',
        'is_active',
        'created_by',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'is_active' => 'boolean',
    ];

    /**
     * Get the parent category.
     */
    public function parent(): BelongsTo
    {
        return $this->belongsTo($this, 'parent_id');
    }

    /**
     * Get the subcategories for this category.
     */
    public function children(): HasMany
    {
        return $this->hasMany($this, 'parent_id');
    }

    /**
     * Get all equipment in this category.
     */
    public function equipments(): BelongsToMany
    {
        return $this->belongsToMany(
            Equipment::class,
            'equipment_equipment_category',
            'equipment_category_id',
            'equipment_id'
        )->withTimestamps();
    }

    /**
     * Get the creator of this category.
     */
    public function author(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Scope a query to only include active categories.
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function getSlugOptions(): SlugOptions
    {
        return SlugOptions::create()
            ->generateSlugsFrom('category_name')
            ->saveSlugsTo('slug');
    }
}
