<?php

namespace App\Models\Job;

use App\Enums\PreferredGender;
use App\Enums\SalaryPeriod;
use App\Enums\JobShift;
use App\Enums\JobWorkMode;
use App\Models\User\User;
use App\Models\Currency;
use App\Models\Location\City;
use App\Models\Location\State;
use App\Models\Location\Country;
use App\Models\Traits\CreatedByTrait;
use App\Models\Traits\HasUuidTrait;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Spatie\Sluggable\HasSlug;
use Spatie\Sluggable\SlugOptions;
use App\Enums\JobStatus;
use App\Observers\JobObserver;
use Illuminate\Support\Number;
use RalphJSmit\Laravel\SEO\Support\HasSEO;
use Illuminate\Database\Eloquent\Attributes\ObservedBy;
use OwenIt\Auditing\Contracts\Auditable;
use OwenIt\Auditing\Auditable as AuditableTrait;
use App\Models\Job\Traits\JobHelperTrait;
use Illuminate\Database\Eloquent\Relations\MorphToMany;

#[ObservedBy(JobObserver::class)]
class Job extends Model implements Auditable
{
    use HasUuidTrait, CreatedByTrait, SoftDeletes, HasFactory, HasSlug, HasSEO, AuditableTrait, JobHelperTrait;

    protected $fillable = [
        'uuid',
        'job_title',
        'slug',
        'job_description',
        'user_id',
        'job_role_id',
        'job_industry_id',
        'available_positions',
        'preferred_gender',
        'min_salary',
        'max_salary',
        'currency_id',
        'salary_period',
        'job_shift',
        'work_mode',
        'apply_by_date',
        'deactivate_apply_by',
        'key_responsibilities',
        'required_qualifications',
        'nice_to_have_skills',
        'bonus_and_benefits',
        'country_id',
        'state_id',
        'city_id',
        'is_featured',
        'is_active',
        'created_by',
        'updated_by',
        'published_at',
        'job_status',
        'expired_at'
    ];

    protected $casts = [
        'published_at' => 'datetime',
        'expired_at' => 'datetime',
        'apply_by_date' => 'date',
        'deactivate_apply_by' => 'boolean',
        'is_featured' => 'boolean',
        'preferred_gender' => PreferredGender::class,
        'salary_period' => SalaryPeriod::class,
        'job_shift' => JobShift::class,
        'work_mode' => JobWorkMode::class,
        'job_status' => JobStatus::class
    ];

    protected $auditInclude = [
        'job_title',
        'slug',
        'job_status',
        'published_at',
    ];


    public function getSlugOptions(): SlugOptions
    {
        return SlugOptions::create()->generateSlugsFrom('job_title')->saveSlugsTo('slug')->doNotGenerateSlugsOnUpdate();
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function job_role(): BelongsTo
    {
        return $this->belongsTo(JobRole::class);
    }

    public function job_industry(): BelongsTo
    {
        return $this->belongsTo(JobIndustry::class);
    }

    public function currency(): BelongsTo
    {
        return $this->belongsTo(Currency::class);
    }

    public function country(): BelongsTo
    {
        return $this->belongsTo(Country::class);
    }

    public function state(): BelongsTo
    {
        return $this->belongsTo(State::class);
    }

    public function city(): BelongsTo
    {
        return $this->belongsTo(City::class);
    }

    public function author(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function job_skills(): BelongsToMany
    {
        return $this->belongsToMany(JobSkill::class, 'job_job_skills');
    }

    public function job_career_level(): BelongsTo
    {
        return $this->belongsTo(JobCareerLevel::class);
    }

    public function job_type(): BelongsTo
    {
        return $this->belongsTo(JobType::class);
    }

    public function applications(): HasMany
    {
        return $this->hasMany(JobApplication::class);
    }

    public function favoritesUsers(): MorphToMany
    {
        return $this->morphToMany(User::class, 'model');
    }

    public function getJobPostedDateAttribute(): string
    {
        return \Carbon\Carbon::parse($this->created_at)->format('F j, Y');
    }

    public function getJobDeadlineDateAttribute(): string
    {
        return \Carbon\Carbon::parse($this->apply_by_date)->format('F j, Y');
    }

    public function getJobSalaryRangeAttribute(): string
    {
        $salery_range = '';
        $min_salery = $this->min_salary ? Number::abbreviate($this->min_salary) : null;
        $max_salery = $this->max_salary ? Number::abbreviate($this->max_salary) : null;

        if (!empty($min_salery)) {
            $salery_range = '$' . $min_salery . ' - ';
        }

        if (!empty($max_salery)) {
            $salery_range = $salery_range . '$' . $max_salery;
        }
        if (!empty($salery_range)) {
            $salery_range = $salery_range . ' Per Year';
        }

        return $salery_range;
    }

    public function getJobIsFeaturedAttribute(): bool
    {
        return $this->is_featured == 1;
    }

    public function getJobEmploymentTypeAttribute(): string
    {
        $employment_type = '';

        if (!empty($this->job_type)) {
            $employment_type = $this->job_type->job_type_name;
        }
        return $employment_type;
    }

    public function getJobRoleNameAttribute(): string
    {
        $job_role = '';
        if (!empty($this->job_role)) {
            $job_role = $this->job_role->job_role_name;
        }
        return $job_role;
    }
    public function getJobCareerLevelNameAttribute(): string
    {
        $career_level = '';
        if (!empty($this->job_career_level)) {
            $career_level = $this->job_career_level->career_level_name;
        }
        return $career_level;
    }

    public function getIsNewAttribute(): bool
    {
        $published_date = \Carbon\Carbon::parse($this->published_at);
        $isNew = $published_date && $published_date->diffInHours(now()) <= 24;
        return $isNew;
    }
}
