<?php

namespace App\Models\Job\Traits;

use Illuminate\Support\Number;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;


trait JobHelperTrait
{
    public function getPermalinkAttribute(): string
    {
        return route('job.show', ['slug' => $this->slug]);
    }

    public function address($type = 'full'): string
    {
        $address = '';

        $addressParts = [];
        if ($this->city) {
            $addressParts['city'] = $this->city->city_name;
        }
        if ($this->state) {
            $addressParts['state'] = $this->state->state_name;
            $addressParts['state_code'] = $this->state->state_code;
        }
        if ($this->country) {
            $addressParts['country'] = $this->country->country_name;
            $addressParts['country_code'] = $this->country->country_code;
        }

        if ($type == 'full') {
            unset($addressParts['country_code']);
            unset($addressParts['state_code']);
            $address = implode(', ', $addressParts);
        } else {
            unset($addressParts['country_code']);
            unset($addressParts['state']);
            if(empty($addressParts['state_code'])){
                unset($addressParts['state_code']);
            }
            $address = implode(', ', $addressParts);
        }
        return $address;
    }

    public function jobTypeName(): string
    {
        $jobType = '';
        if (! empty($this->job_type)) {
            $jobType = $this->job_type->job_type_name;
        }
        return $jobType;
    }

    public function saleryRange(): string
    {
        $saleryRange = '';
        $minSalary   = $this?->min_salary ? Number::abbreviate($this?->min_salary) : '';
        $maxSalary   = $this?->max_salary ? Number::abbreviate($this?->max_salary) : '';

        if (! empty($minSalary)) {
            $saleryRange = '$' . $minSalary . '/yr';
        }
        if (empty($minSalary) && ! empty($maxSalary)) {
            $saleryRange = '$' . $maxSalary . '/yr';
        }
        if (! empty($minSalary) && ! empty($maxSalary)) {
            $saleryRange = '$' . $minSalary . '/yr - $' . $maxSalary . '/yr';
        }

        return $saleryRange;
    }

    public function jobCarerLevel(): string
    {
        $jobCarerLevel = '';
        if (! empty($this->job_career_level)) {
            $jobCarerLevel = $this->job_career_level->career_level_name;
        }
        return $jobCarerLevel;
    }
    public function jobIndustry(): string
    {
        $jobIndustry = '';
        if (! empty($this->job_industry)) {
            $jobIndustry = $this->job_industry->industry_name;
        }
        return $jobIndustry;
    }
    public function jobRole(): string
    {
        $jobRole = '';
        if (!empty($this->job_role)) {
            $jobRole = $this->job_role->job_role_name;
        }
        return $jobRole;
    }

    public function postedDate(): string
    {
        $postedDate = '';
        if (!empty($this->created_at)) {
            $postedDate = $this->created_at->diffForHumans();
        }
        return $postedDate;
    }

    public function deadlineDate(): string
    {
        $deadlineDate = '';
        if (!empty($this->apply_by_date)) {
            $deadlineDate = \Carbon\Carbon::parse($this->apply_by_date)->format('M j, Y');
        }
        return $deadlineDate;
    }

    public function isNew(): bool
    {
        $isNew = false;
        if (!empty($this->published_at)) {
            $published_date = \Carbon\Carbon::parse($this->published_at);
            $isNew = $published_date->diffInHours(now()) <= 24;
        }

        return $isNew;
    }

    public function workMode(): string
    {
        $workMode = '';
        if (!empty($this->work_mode)) {
            $workMode = $this->work_mode->getLabel();
        }
        return $workMode;
    }

    public function authorName(): string
    {
        $authorName = '';
        if (!empty($this->user)) {
            $authorName = $this->author->profile()->exists() ? $this->author->profile->company_name ?? $this->author->name : $this->author->name;
        }
        return $authorName;
    }

    public function mapIframeUrl($settings = []): ?string
    {
        $zoomLevel = $settings['zoom_level'] ?? 15;
        $mapType   = $settings['map_type'] ?? 'roadmap';

        $country = $this->country;
        $state = $this->state;
        $city = $this->city;

        if (!$country) {
            return null;
        }

        $location = [];
        if ($city && $city->city_name) {
            $location[] = $city->city_name;
        }
        if ($state && $state->state_name) {
            $location[] = $state->state_name;
        }
        if ($country && $country->country_name) {
            $location[] = $country->country_name;
        }

        if (empty($location)) {
            return null;
        }

        $coords = implode(', ', $location);
        $mapUrl = 'https://www.google.com/maps/embed/v1/place?key=' . config('services.google.maps_api_key', '') .
            "&q=" . urlencode($coords) .
            '&zoom=' . $zoomLevel .
            '&maptype=' . $mapType;

        return $mapUrl;
    }

    public function isFavorite(): bool
    {
        if (!Auth::check()) {
            return false;
        }
        return Auth::user()->favoriteJob()->where('model_id', $this->id)->exists();
    }

    public function isApplied(): bool
    {
        if (!Auth::check()) {
            return false;
        }
        return Auth::user()->jobApplications()->where('job_id', $this->id)->exists();
    }
}
