<?php

namespace App\Models\Lead;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Spatie\Image\Enums\Fit;
use Spatie\MediaLibrary\HasMedia;
use Spatie\MediaLibrary\InteractsWithMedia;
use Spatie\MediaLibrary\MediaCollections\Models\Media;

class FieldLeadForm extends Model implements HasMedia
{
    use HasFactory, InteractsWithMedia;

    protected $table = 'field_lead_form';

    protected $fillable = [
        'uuid',
        'lead_form_id',
        'field_id',
        'field_label',
        'field_placeholder',
        'toggle_yes_text',
        'toggle_no_text',
        'field_columns',
        'help_text',
        'data_source',
        'field_options',
        'field_data',
        'conditional_logic',
        'is_required',
        'is_visible',
        'field_order',
        'form_step',
    ];

    protected $casts = [
        'field_data' => 'array',
        'conditional_logic' => 'array',
        'is_required' => 'boolean',
        'is_visible' => 'boolean',
    ];

    /**
     * Get the field_options attribute
     * Don't cast to array if it's an enum class path
     */
    protected function fieldOptions(): \Illuminate\Database\Eloquent\Casts\Attribute
    {
        return \Illuminate\Database\Eloquent\Casts\Attribute::make(
            get: function ($value) {
                if ($value === null) {
                    return null;
                }

                // If data_source is 'enum', keep it as a string (enum class path)
                if ($this->data_source === 'enum') {
                    return $value;
                }

                // Otherwise, try to decode as JSON array
                $decoded = json_decode($value, true);

                return is_array($decoded) ? $decoded : $value;
            },
            set: function ($value) {
                if ($value === null) {
                    return null;
                }

                // If data_source is 'enum', store as plain string
                if ($this->data_source === 'enum') {
                    return is_string($value) ? $value : json_encode($value);
                }

                // Otherwise, store as JSON if it's an array
                return is_array($value) ? json_encode($value) : $value;
            }
        );
    }

    public function leadForm(): BelongsTo
    {
        return $this->belongsTo(LeadForm::class);
    }

    public function field(): BelongsTo
    {
        return $this->belongsTo(LeadFormFields::class, 'field_id');
    }

    /**
     * Get the parent field this field depends on
     */
    public function parentField(): BelongsTo
    {
        return $this->belongsTo(self::class, 'conditional_logic->parent_field_id', 'id');
    }

    /**
     * Get all child fields that depend on this field
     */
    public function childFields(): HasMany
    {
        return $this->hasMany(self::class, 'conditional_logic->parent_field_id', 'id');
    }

    /**
     * Check if this field has conditional logic (is a dependent field)
     */
    public function isConditional(): bool
    {
        return ! empty($this->conditional_logic['enabled']) && ! empty($this->conditional_logic['parent_field_id']);
    }

    /**
     * Get the parent field ID from conditional logic
     */
    public function getParentFieldId(): ?int
    {
        return $this->conditional_logic['parent_field_id'] ?? null;
    }

    /**
     * Get the parent option value that triggers this field
     */
    public function getParentValue(): ?string
    {
        return $this->conditional_logic['parent_option_value'] ?? null;
    }

    /**
     * Get the maximum nesting level for this field's dependencies
     */
    public function getDependencyLevel(int $currentLevel = 0): int
    {
        if ($currentLevel >= 3 || ! $this->isConditional()) {
            return $currentLevel;
        }

        $parent = self::find($this->getParentFieldId());

        if (! $parent) {
            return $currentLevel;
        }

        return $parent->getDependencyLevel($currentLevel + 1);
    }

    /**
     * Register the media collections for the model.
     */
    public function registerMediaCollections(): void
    {
        $this->addMediaCollection('section_logo')
            ->singleFile()
            ->useFallbackUrl('/images/logo_placeholder.png')
            ->useFallbackPath(public_path('images/logo_placeholder.png'))
            ->registerMediaConversions(function (Media $media) {
                $this->addMediaConversion('thumb')
                    ->fit(Fit::Contain, 200)
                    ->keepOriginalImageFormat()
                    ->sharpen(10);
            });
    }
}
