<?php

namespace App\Models\Lead;

use App\Enums\LeadFormLayout;
use App\Enums\LeadFormMatchCriteria;
use App\Enums\LeadFormReceptionMode;
use App\Models\Traits\CreatedByTrait;
use App\Models\Traits\HasUuidTrait;
use App\Models\User\MemberCategory;
use App\Models\User\User;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;

class LeadForm extends Model
{
    use CreatedByTrait, HasFactory, HasUuidTrait, SoftDeletes;

    protected $fillable = [
        'uuid',
        'title',
        'description',
        'match_criteria',
        'reception_mode',
        'enable_captcha',
        'is_active',
        'submit_message',
        'form_layout',
        'form_steps',
        'created_by',
    ];

    protected $casts = [
        'fields' => 'array',
        'form_steps' => 'array',
        'form_layout' => LeadFormLayout::class,
        'match_criteria' => LeadFormMatchCriteria::class,
        'reception_mode' => LeadFormReceptionMode::class,
    ];

    public function visits(): HasMany
    {
        return $this->hasMany(LeadFormVisit::class);
    }

    public function leads(): HasMany
    {
        return $this->hasMany(Lead::class);
    }

    public function fieldLeadForm(): HasMany
    {
        return $this->hasMany(FieldLeadForm::class);
    }

    public function formFields(): BelongsToMany
    {
        return $this->belongsToMany(LeadFormFields::class, 'field_lead_form', 'lead_form_id', 'field_id')
            ->withPivot([
                'id',
                'field_label',
                'field_placeholder',
                'toggle_yes_text',
                'toggle_no_text',
                'field_columns',
                'help_text',
                'data_source',
                'field_options',
                'field_data',
                'conditional_logic',
                'is_required',
                'is_visible',
                'field_order',
                'form_step',
            ])->orderByPivot('field_order');
    }

    public function author(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function memberCategories(): BelongsToMany
    {
        return $this->belongsToMany(MemberCategory::class)
            ->withPivot('credits', 'radius');
    }
}
